/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.colorator;

import kandid.genomedb.ChromosomeType;

/**
 *
 * @author thomas jourdan
 * @version
 */
public class RGBColorator implements Colorator {

  /**
   * Convert red / green / blue to RGB color.
   *
   * @param redValue a red component, value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param greenValue a green component, value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param blueValue a blue component, value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double redValue, double greenValue, double blueValue, ColorF32 cout) {
    assert cout != null;
    if (redValue >= 1.0) {
      cout.r = 1.0f;
    }
    else if (redValue < 0.0) {
      cout.r = 0.0f;
    }
    else {
      cout.r = (float)redValue;
    }
    
    if (greenValue >= 1.0) {
      cout.g = 1.0f;
    }
    else if (greenValue < 0.0) {
      cout.g = 0.0f;
    }
    else {
      cout.g = (float)greenValue;
    }
    
    if (blueValue >= 1.0) {
      cout.b = 1.0f;
    }
    else if (blueValue < 0.0) {
      cout.b = 0.0f;
    }
    else {
      cout.b = (float)blueValue;
    }
  }
  
  /**
   * Convert value to gray scale
   *
   * @param value a value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double value, ColorF32 cout) {
    assert cout != null;
    float gray;
    if (value > 1.0) {
      gray = 1.0f;
    }
    else if (value < 0.0) {
      gray = 0.0f;
    }
    else {
      gray = (float)value;
    }
    cout.r = gray;
    cout.g = gray;
    cout.b = gray;
  }

  /**
   * Method getColor.
   * @param colorGene
   */
  public void getColor(kandid.soup.ColorGene colorGene, ColorF32 cout) {
    getColor(colorGene.getRed(), colorGene.getGreen(), colorGene.getBlue(), cout);
  }

  /**
   * Method getColor.
   * @param color
   */
  public void getColor(java.awt.Color color, ColorF32 cout) {
    getColor(color.getRed() / 256.0, color.getGreen() / 256.0, color.getBlue() / 256.0, cout);
  }

  /**
   * Method getShortName.
   * @return String
   */
  public String getShortName() {
    return "RGB";
  }

  /**
   * not implemented. Makes no sense in this context
   * @see kandid.colorator.Colorator#setColorTable(ChromosomeType)
   */
  public void setColorTable(kandid.soup.ChromosomeType colorTable) {
    // don't implement
  }

  /**
   * not implemented. Makes no sense in this context
   * @see kandid.colorator.Colorator#getDiscretColor(int, ColorF32)
   */
  public void getDiscretColor(int index, ColorF32 cout) {
    // don't implement
    assert cout != null;
    cout.r = 0.0f;
    cout.g = 0.0f;
    cout.b = 0.0f;
  }

  /**
   * not implemented. Makes no sense in this context
   * @see kandid.colorator.Colorator#getNumberOfDiscretColors(int)
   */
  public int getNumberOfDiscretColors() {
    return 0;
  }

}
