/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.colorator;

import kandid.soup.ChromosomeType;
import kandid.soup.GradientChromosome;
import kandid.soup.GradientSegmentGene;

/**
 *
 * @author thomas jourdan
 */
public class GradientColorator implements Colorator {
  private float hueSegment[];
  private float saturationSegment[];
  private float brightnessSegment[];
  private double segmentStart[];
  private int    size;

  /**
   * Uses value as an index to the color table
   *
   * @param value a value between 0.0 (inclusive) and 1.0 (inclusive).
   */
  public void getColor(double value, ColorF32 cout) {
    assert cout != null;
    if(value <= 0.0) {
      ColorF32 hsb = new ColorF32(hueSegment[0], saturationSegment[0], brightnessSegment[0]);
      ColorF32.hsb2rgb(hsb, cout);
      return;
    }
    if(value >= 1.0) {
      ColorF32 hsb = new ColorF32(hueSegment[size-1], saturationSegment[size-1], brightnessSegment[size-1]);
      ColorF32.hsb2rgb(hsb, cout);
      return;
    }
    int upperSegment = 1;
    while(segmentStart[upperSegment] < value && upperSegment < size) {
      ++upperSegment;
    }
    int lowerSegment = upperSegment-1;
    double v = (segmentStart[upperSegment] - value) / (segmentStart[upperSegment] - segmentStart[lowerSegment]);
    ColorF32 hsb = new ColorF32(interpolateHue(hueSegment[lowerSegment],     hueSegment[upperSegment],        v),
                                interpolate(saturationSegment[lowerSegment], saturationSegment[upperSegment], v),
                                interpolate(brightnessSegment[lowerSegment], brightnessSegment[upperSegment], v));
    ColorF32.hsb2rgb(hsb, cout);
  }

  /**
   * Method interpolate.
   * @param lowerValue
   * @param upperValue
   * @param v
   * @return float
   */
  public static float interpolate(float lowerValue, float upperValue, double v) {
    return (float)(lowerValue * v + upperValue * (1.0 - v));
  } 


  /**
   * Method interpolate.
   * @param lowerValue
   * @param upperValue
   * @param v
   * @return float
   */
  public static float interpolateHue(float lowerValue, float upperValue, double v) {
    double diff = LookUpTableColorator.diffHue(lowerValue, upperValue);
    if(LookUpTableColorator.isMinDiffCW(lowerValue, upperValue)) {
      double diff1 = lowerValue + (1.0-v) * diff;
      return (float)(diff1-Math.floor(diff1));
    }
    else {
      double diff1 = upperValue + (1.0-v) * diff;
      return (float)(diff1-Math.floor(diff1));
    }
  } 


  /**
   * Convert red / green / blue to gray scale.
   *
   * @param redValue a value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param greenValue a value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param blueValue a value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double redValue, double greenValue, double blueValue, ColorF32 cout) {
    assert cout != null;
    double gray = 0.30 * redValue + 0.59 * greenValue + 0.11 * blueValue;
    if (gray >= 1.0) {
      gray = 1.0f;
    }
    else if (gray < 0.0) {
      gray = 0.0f;
    }
    cout.r = (float) gray;
    cout.g = (float) gray;
    cout.b = (float) gray;
  }
  
  /**
   * Method getColor.
   * @param colorGene
   */
  public void getColor(kandid.soup.ColorGene colorGene, ColorF32 cout) {
    assert cout != null;
    cout.r = (float)colorGene.getRed();
    cout.g = (float)colorGene.getGreen();
    cout.b = (float)colorGene.getBlue();
  }

  /**
   * Method getColor.
   * @param color
   */
  public void getColor(java.awt.Color color, ColorF32 cout) {
    assert cout != null;
    cout.r = (float)(color.getRed() / 256.0);
    cout.g = (float)(color.getGreen() / 256.0);
    cout.b = (float)(color.getBlue() / 256.0);
  }
  
  /**
   * Method getShortName.
   * @return String
   */
  public String getShortName() {
    return "Grad";
  }

  /**
   * @see kandid.colorator.Colorator#setColorTable(ChromosomeType)
   */
  public void setColorTable(kandid.soup.ChromosomeType colorTable) {
    size = ((GradientChromosome)colorTable).getGradientSegment().size();
    segmentStart      = new double[size];
    hueSegment        = new float[size];
    saturationSegment = new float[size];
    brightnessSegment = new float[size];
    GradientSegmentGene gradientSegment[] = new GradientSegmentGene[size];
    ((GradientChromosome)colorTable).getGradientSegment().toArray(gradientSegment);
    double sum = 0.0;
    for (int ix = 0; ix < size; ++ix) {
      segmentStart[ix] = sum;
      if((ix+1) < segmentStart.length) {
        sum += gradientSegment[ix].getWeight().getValue();
      }
      hueSegment[ix]        = (float)(double)gradientSegment[ix].getHue().getValue();
      saturationSegment[ix] = (float)(double)gradientSegment[ix].getSaturation().getValue();
      brightnessSegment[ix] = (float)(double)gradientSegment[ix].getBrightness().getValue();
    }
    // normalize
    for (int ix = 0; ix < size; ++ix) {
      segmentStart[ix] /= sum;
    }
  }

  /**
   * not implemented. Makes no sense in this context
   * @see kandid.colorator.Colorator#getDiscretColor(int, ColorF32)
   */
  public void getDiscretColor(int index, ColorF32 cout) {
    // don't implement
    assert cout != null;
    cout.r = 0.0f;
    cout.g = 0.0f;
    cout.b = 0.0f;
  }

  /**
   * not implemented. Makes no sense in this context
   * @see kandid.colorator.Colorator#getNumberOfDiscretColors(int)
   */
  public int getNumberOfDiscretColors() {
    // don't implement
    return 0;
  }

}
