/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.colorator;

import kandid.soup.ColoratorType;
import kandid.util.Debug;

public class ColoratorFactory {

  /**
   * Tests if the colorator has no chromosomes.
   * @return true if the colorator has no chromosomes.
   */
  public static boolean isSimpleColorator(String coloratorName) {
    return coloratorName.endsWith(".BlackWhiteColorator")
           || coloratorName.endsWith(".GrayColorator")
           || coloratorName.endsWith(".RGBColorator");
  }

  /**
   * Method createColoratorWorker.
   * @param coloratorName
   * @return ColoratorWorker
   */
  public static Colorator createColoratorWorker(String coloratorName, ColoratorType coloratorType) {
    if(Debug.enabled) assert coloratorName.startsWith("kandid.colorator."): "check colorator name: " + coloratorName;
    try {
      Class cc = Class.forName(coloratorName);
      Colorator colorator = (Colorator)cc.getDeclaredConstructor().newInstance();
      if(!isSimpleColorator(coloratorName)) {
        if(coloratorName.equals("kandid.colorator.LookUpTableColorator")) {
          colorator.setColorTable(coloratorType.getLookUpTable());
        }
        else if(coloratorName.equals("kandid.colorator.TransparentLookUpTableColorator")) {
          colorator.setColorTable(coloratorType.getTransparentLookUpTable());
        }
        else if(coloratorName.equals("kandid.colorator.GradientColorator")) {
          colorator.setColorTable(coloratorType.getGradient());
        }
        else if(coloratorName.equals("kandid.colorator.HsbFrequenceColorator")) {
          colorator.setColorTable(coloratorType.getHsbFrequence());
        }
      }
      return colorator;
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return null;
  }

  /**
   * Method getShortName.
   * @param coloratorName
   * @param colorator
   * @return String
   */
  public static String getShortName(String coloratorName) {
    if(Debug.enabled) assert coloratorName.startsWith("kandid.colorator."): "check colorator name: " + coloratorName;
    try {
      Class cc = Class.forName(coloratorName);
      Colorator colorator = (Colorator)cc.getDeclaredConstructor().newInstance();
      return colorator.getShortName();
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
    return "";
  }

}
