/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.colorator;

/**
 *
 * @author thomas jourdan
 */
public class BlackWhiteColorator implements Colorator {

  /**
   * Convert value to black and white.
   *
   * @param value a value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double value, ColorF32 cout) {
    assert cout != null;
    if(value > 0.5) {
      cout.r = 1.0f;
      cout.g = 1.0f;
      cout.b = 1.0f;
    }
    else {
      cout.r = 0.0f;
      cout.g = 0.0f;
      cout.b = 0.0f;
    }
  }

  /**
   * Convert red / green / blue to black and white.
   *
   * @param redValue a red component, value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param greenValue a green component, value between 0.0 (inclusive) and 1.0 (exclusive).
   * @param blueValue a blue component, value between 0.0 (inclusive) and 1.0 (exclusive).
   */
  public void getColor(double redValue, double greenValue, double blueValue, ColorF32 cout) {
    assert cout != null;
    if((0.30 * redValue + 0.59 * greenValue + 0.11 * blueValue) > 0.5) {
      cout.r = 1.0f;
      cout.g = 1.0f;
      cout.b = 1.0f;
    }
    else {
      cout.r = 0.0f;
      cout.g = 0.0f;
      cout.b = 0.0f;
    }
  }
  
  /**
   * Method getColor.
   * @param colorGene
   */
  public void getColor(kandid.soup.ColorGene colorGene, ColorF32 cout) {
    getColor(colorGene.getRed(), colorGene.getGreen(), colorGene.getBlue(), cout);
  }

  /**
   * Method getColor.
   * @param color
   */
  public void getColor(java.awt.Color color, ColorF32 cout) {
    getColor(color.getRed() / 256.0, color.getGreen() / 256.0, color.getBlue() / 256.0, cout);
  }

  /**
   * Method getShortName.
   * @return String
   */
  public String getShortName() {
    return "BlackWhite";
  }

  /**
   * not implemented. Makes no sense in this context
   * @see kandid.colorator.Colorator#setColorTable(ChromosomeType)
   */
  public void setColorTable(kandid.soup.ChromosomeType colorTable) {
    // don't implement
  }

  /**
   * Get color value.
   *
   * @param index an index between 0 and colorTable.length-1.
   */
  public void getDiscretColor(int index, ColorF32 cout) {
    assert cout != null;
    if(index > 0) {
      cout.r = 1.0f;
      cout.g = 1.0f;
      cout.b = 1.0f;
    }
    else {
      cout.r = 0.0f;
      cout.g = 0.0f;
      cout.b = 0.0f;
    }
  }

  /**
   * Get number of entries.
   *
   * @return 2
   */
  public int getNumberOfDiscretColors() {
    return 2;
  }

}