/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.vm.vector;

import java.awt.Component;
import java.awt.Dimension;

import javax.vecmath.Vector3d;

import kandid.calculation.RefinementCalculation;
import kandid.calculation.vm.scalar.ScalarVM;
import kandid.colorator.ColorF32;
import kandid.soup.ChromosomeType;
import kandid.soup.VectorExprGene;
import kandid.soup.VectorExpressionChromosome;

public class VectorExpressionCalculation extends RefinementCalculation {

	private ScanResults scanResults;
	private VectorVM vectorVM;

	private class ScanResults {
		boolean isNaN = false;
    double min;
		double range = 0.0001;

		public ScanResults(VectorExpressionChromosome vectorExpressionChromosome) {
			Vector3d vectorValue = new Vector3d();
			Vector3d maxValue = new Vector3d(Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY);
			Vector3d minValue = new Vector3d(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);

			final long nan = Double.doubleToLongBits(Double.NaN);
			vectorVM.execute(0, 0, ScalarVM.epsilon, vectorValue);
			if (nan == Double.doubleToLongBits(vectorValue.x)) {
				isNaN = true;
			}
			else {
				maxValue.x = minValue.x = vectorValue.x;
			}
			if (nan == Double.doubleToLongBits(vectorValue.y)) {
				isNaN = true;
			}
			else {
				maxValue.y = minValue.y = vectorValue.y;
			}
			if (nan == Double.doubleToLongBits(vectorValue.z)) {
				isNaN = true;
			}
			else {
				maxValue.z = minValue.z = vectorValue.z;
			}

			double xt = xwMin;
			while (xt < xwMax + ScalarVM.epsilon) {
				double yt = ywMin;
				while (yt < ywMax + ScalarVM.epsilon) {
					vectorVM.execute(xt, yt, ScalarVM.epsilon, vectorValue);
					if (nan == Double.doubleToLongBits(vectorValue.x)) {
						isNaN = true;
					}
					else {
						if (vectorValue.x < minValue.x)
							minValue.x = vectorValue.x;
						if (vectorValue.x > maxValue.x)
							maxValue.x = vectorValue.x;
					}
					if (nan == Double.doubleToLongBits(vectorValue.y)) {
						isNaN = true;
					}
					else {
						if (vectorValue.y < minValue.y)
							minValue.y = vectorValue.y;
						if (vectorValue.y > maxValue.y)
							maxValue.y = vectorValue.y;
					}
					if (nan == Double.doubleToLongBits(vectorValue.z)) {
						isNaN = true;
					}
					else {
						if (vectorValue.z < minValue.z)
							minValue.z = vectorValue.z;
						if (vectorValue.z > maxValue.z)
							maxValue.z = vectorValue.z;
					}
					yt += 0.1;
				}
				xt += 0.1;
			}

//			double xDiff = maxValue.x - minValue.x;
//			range = xDiff;
//			double yDiff = maxValue.y - minValue.y;
//			if (yDiff > range)
//				range = yDiff;
//			double zDiff = maxValue.z - minValue.z;
//			if (zDiff > range)
//				range = zDiff;
//			if (range < 0.0001)
//				range = 0.0001;

      min = minValue.x;
      if (minValue.y < min)
        min = minValue.y;
      if (minValue.z < min)
        min = minValue.z;
      double max;
      max = maxValue.x;
      if (maxValue.y > max)
        max = maxValue.y;
      if (maxValue.z > max)
        max = maxValue.z;
      range = max - min;
      if (range < 0.0001)
        range = 0.0001;
		}

	}

	/* For documentation please see super class RefinementCalculation. */
	public boolean reject(ChromosomeType chromosome) {
		depth = 6;
		xwMin = -1.0;
		xwMax = 1.0;
		ywMin = -1.0;
		ywMax = 1.0;
		this.chromosome = chromosome;
		VectorExpressionChromosome vectorExpressionChromosome = (VectorExpressionChromosome) chromosome;
		VectorExprGene vectorExprGene = vectorExpressionChromosome.getVectorExpression();
		vectorVM = new VectorVM(vectorExpressionChromosome.getNoiseSeed().getValue());
		try {
      vectorVM.compile(vectorExprGene);
    } catch (Exception e) {
      e.printStackTrace();
      return true;
    }

		scanResults = new ScanResults(vectorExpressionChromosome);
		//!!    return scanResults.isNaN || (scanResults.maxValue - scanResults.minValue) < 0.00009;
		return scanResults.isNaN || scanResults.range <= (0.0001+ScalarVM.epsilon);
	}

	/* For documentation please see super class RefinementCalculation. */
	public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
		depth = 6;
		xwMin = -1.1;
		xwMax = 1.1;
		ywMin = -1.1;
		ywMax = 1.1;
		super.activateCanvas(viewComponent, canvasSize, zoomMode);

		VectorExpressionChromosome vectorExpressionChromosome = (VectorExpressionChromosome) chromosome;
		VectorExprGene vectorExprGene = vectorExpressionChromosome.getVectorExpression();
		vectorVM = new VectorVM(vectorExpressionChromosome.getNoiseSeed().getValue());
		vectorVM.compile(vectorExprGene);

		scanResults = new ScanResults((VectorExpressionChromosome) chromosome);
	}

	/* (non-Javadoc)
	 * @see kandid.calculation.Calculation#calculate(boolean)
	 */
	public void calculate(boolean paintOnScreen, String doExport) {
		if (enterDepth()) {
			Vector3d vectorValue = new Vector3d();
	    ColorF32 cout = new ColorF32();
			while (more) {
				vectorVM.execute(xw, yw, ScalarVM.epsilon, vectorValue);
				colorator.getColor(
            (vectorValue.x - scanResults.min) / scanResults.range,
            (vectorValue.y - scanResults.min) / scanResults.range,
            (vectorValue.z - scanResults.min) / scanResults.range, cout);
        setPixel(cout);
//				float h = (float) ((vectorValue.x - scanResults.min) / scanResults.range);
//				float s = (float) ((vectorValue.y - scanResults.min) / scanResults.range);
//				float b = (float) ((vectorValue.z - scanResults.min) / scanResults.range);
//        assert h > -0.01f && h < 1.01f : "h = " + h + "out of range";
//        assert s > -0.01f && s < 1.01f : "s = " + s + "out of range";
//        assert b > -0.01f && b < 1.01f : "b = " + b + "out of range";
//        if(h < 0.0f)
//          h = 0.0f;
//        else if(h > 1.0f)
//          h = 1.0f;
//        if(s < 0.0f)
//          s = 0.0f;
//        else if(s > 1.0f)
//          s = 1.0f;
//        if(b < 0.0f)
//          b = 0.0f;
//        else if(b > 1.0f)
//          b = 1.0f;
         
//!!        setPixel(colorator.getColor(Color.HSBtoRGB(h, s, b)));
//      setPixel(Color.HSBtoRGB(h, s, b));
				nextLocation();
			}
		}
	}

	/* (non-Javadoc)
	 * @see kandid.calculation.Calculation#getBackgroundColor()
	 */
	public boolean hasWhiteBackground() {
		return false;
	}

}
