/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.vm.scalar;

import junit.framework.TestCase;
import kandid.soup.ScalarExprGene;
import kandid.soup.util.SoupFactory;

/**
 * 
 * @author thomas jourdan
 */
public class ScalarVMTest extends TestCase {

  private ScalarVM scalarVM;
  private SoupFactory soupFactory;

  /**
   * Constructor for ScalarVMTest.
   * @param arg0
   */
  public ScalarVMTest(String arg0) {
    super(arg0);
  }

  /**
    * @see junit.framework.TestCase#setUp()
    */
  protected void setUp() throws Exception {
    super.setUp();
    scalarVM = new ScalarVM(151258);
    soupFactory = SoupFactory.getSoupFactory();
  }

  private void compileExpr(String expr) {
    kandid.soup.EntityType entity = (kandid.soup.EntityType) soupFactory.unmarshalFromString(expr);
    ScalarExprGene scalarExprGene = entity.getScalarExpression().getChromosome().getScalarExpression();
    System.out.println("----\n");
    scalarVM.compile(scalarExprGene);
    System.out.println(""+scalarVM);
  }

  private double executeLispExpr(String expr, double x, double y) {
    compileExpr((new ScalarLisp()).getAsXML(expr));
    return scalarVM.execute(x, y, ScalarVM.epsilon);
  }

  private String trimLines(String expr) {
    char[] inp = expr.toCharArray();
    StringBuffer trimmed = new StringBuffer();
    for (int ix = 0; ix < inp.length; ix++) {
      if(inp[ix] != ' ') {
        trimmed.append(inp[ix]);
      }
    }
    return trimmed.toString();
  }

  String expr1 =
  HeaderFooter.header
      + "<scalarExpression>\n"
      + "    <list>\n"
      + "        <opr>\n"
      + "            <scalarOpr>sub</scalarOpr>\n"
      + "        </opr>\n"
      + "        <args>\n"
      + "            <const>\n"
      + "                <scalarConst>\n"
      + "                    <value>0.7</value>\n"
      + "                </scalarConst>\n"
      + "            </const>\n"
      + "            <var>\n"
      + "                <scalarVar>x</scalarVar>\n"
      + "            </var>\n"
      + "        </args>\n"
      + "    </list>\n"
      + "</scalarExpression>\n"
      + HeaderFooter.footer;

  public void ignore_testCompile1() {
    compileExpr(expr1);
    int ip = 0;

    assertEquals(ScalarVM.oprPushX, scalarVM.codeTape[ip]);
    assertEquals(0.0, scalarVM.dOper[ip], ScalarVM.epsilon);
    assertEquals(0, scalarVM.iOper[ip]);
    ++ip;
    assertEquals(ScalarVM.oprPushConst, scalarVM.codeTape[ip]);
    assertEquals(0.7, scalarVM.dOper[ip], ScalarVM.epsilon);
    assertEquals(0, scalarVM.iOper[ip]);
    ++ip;
    assertEquals(ScalarVM.oprSub, scalarVM.codeTape[ip]);
    assertEquals(0.0, scalarVM.dOper[ip], ScalarVM.epsilon);
    assertEquals(2, scalarVM.iOper[ip]);
    ++ip;
    assertEquals(0, scalarVM.codeTape[ip]);
    assertEquals(0.0, scalarVM.dOper[ip], ScalarVM.epsilon);
    assertEquals(0, scalarVM.iOper[ip]);
  }

  public void ignore_testExecute1() {
    ignore_testCompile1();
    assertEquals(-0.3, scalarVM.execute(1.0, -1.0, ScalarVM.epsilon), ScalarVM.epsilon);
  }

  String expr2 =
  HeaderFooter.header
      + "<scalarExpression>\n"
      + "    <list>\n"
      + "        <opr>\n"
      + "            <scalarOpr>div</scalarOpr>\n"
      + "        </opr>\n"
      + "        <args>\n"
      + "            <scalarExpression>\n"
      + "                <list>\n"
      + "                    <opr>\n"
      + "                        <scalarOpr>mult</scalarOpr>\n"
      + "                    </opr>\n"
      + "                    <args>\n"
      + "                        <var>\n"
      + "                            <scalarVar>y</scalarVar>\n"
      + "                        </var>\n"
      + "                        <const>\n"
      + "                            <scalarConst>\n"
      + "                                <value>3</value>\n"
      + "                            </scalarConst>\n"
      + "                        </const>\n"
      + "                        <var>\n"
      + "                            <scalarVar>x</scalarVar>\n"
      + "                        </var>\n"
      + "                    </args>\n"
      + "                </list>\n"
      + "            </scalarExpression>\n"
      + "            <const>\n"
      + "                <scalarConst>\n"
      + "                    <value>4</value>\n"
      + "                </scalarConst>\n"
      + "            </const>\n"
      + "            <var>\n"
      + "                <scalarVar>x</scalarVar>\n"
      + "            </var>\n"
      + "        </args>\n"
      + "    </list>\n"
      + "</scalarExpression>\n"
      + HeaderFooter.footer;

  public void ignore_testExecute2() {
    compileExpr(expr2);
    assertEquals(0.75, scalarVM.execute(2.0, 1.0, ScalarVM.epsilon), ScalarVM.epsilon);
  }

  String exprIfPositive =
  HeaderFooter.header
      + "<scalarExpression>\n"
      + "    <list>\n"
      + "        <opr>\n"
      + "            <scalarOpr>ifPositive</scalarOpr>\n"
      + "        </opr>\n"
      + "        <args>\n"
      + "            <var>\n"
      + "                <scalarVar>y</scalarVar>\n"
      + "            </var>\n"
      + "            <const>\n"
      + "                <scalarConst>\n"
      + "                    <value>0.3</value>\n"
      + "                </scalarConst>\n"
      + "            </const>\n"
      + "            <var>\n"
      + "                <scalarVar>x</scalarVar>\n"
      + "            </var>\n"
      + "        </args>\n"
      + "    </list>\n"
      + "</scalarExpression>\n"
      + HeaderFooter.footer;

  public void ignore_testCompileIfPositive() {
    compileExpr(exprIfPositive);
  }

  public void ignore_testExecuteIfPositive() {
    compileExpr(exprIfPositive);
    assertEquals(0.3, scalarVM.execute(0.2, ScalarVM.epsilon, ScalarVM.epsilon), ScalarVM.epsilon);
  }

  String exprNoise =
  HeaderFooter.header
      + "<scalarExpression>\n"
      + "    <list>\n"
      + "        <opr>\n"
      + "            <scalarOpr>noise</scalarOpr>\n"
      + "        </opr>\n"
      + "        <args>\n"
      + "            <const>\n"
      + "                <scalarConst>\n"
      + "                    <value>0.1</value>\n"
      + "                </scalarConst>\n"
      + "            </const>\n"
      + "            <var>\n"
      + "                <scalarVar>x</scalarVar>\n"
      + "            </var>\n"
      + "            <var>\n"
      + "                <scalarVar>y</scalarVar>\n"
      + "            </var>\n"
      + "        </args>\n"
      + "    </list>\n"
      + "</scalarExpression>\n"
      + HeaderFooter.footer;

  public void ignore_testExecuteNoise() {
    compileExpr(exprNoise);
    assertEquals(0.18026221874439646, scalarVM.execute(0.2, 0.1, ScalarVM.epsilon), ScalarVM.epsilon);
  }

  public void testGetAsXML() {
    assertEquals(trimLines(expr1), trimLines((new ScalarLisp()).getAsXML("(sub 0.7 x)")));
    assertEquals(trimLines(expr2), trimLines((new ScalarLisp()).getAsXML("(div (mult y 3 x) 4 x)")));
    assertEquals(trimLines(exprIfPositive), trimLines((new ScalarLisp()).getAsXML("(ifPositive y 0.3 x)")));
    assertEquals(trimLines(exprNoise), trimLines((new ScalarLisp()).getAsXML("(noise 0.1 x y)")));
  }

  public void ignore_testExecuteOpr() {
    assertEquals(Double.POSITIVE_INFINITY,  executeLispExpr("(div x y)", 1, 0), ScalarVM.epsilon);
    assertEquals(1, executeLispExpr("(ifPositive x 1 2)", Double.POSITIVE_INFINITY, 0), ScalarVM.epsilon);
    assertEquals(Double.NEGATIVE_INFINITY,  executeLispExpr("(sub 0 x)", Double.POSITIVE_INFINITY, 0), ScalarVM.epsilon);
    assertEquals(0.5462180318903666,  executeLispExpr("(turbulence 0.1 0.2 0.3 x)", Double.POSITIVE_INFINITY, 0), ScalarVM.epsilon);
    assertEquals(Double.doubleToLongBits(Double.NaN),  Double.doubleToLongBits(executeLispExpr("(sub 1 x)", Double.NaN, 0)));
    assertEquals(Double.doubleToLongBits(Double.NaN),  Double.doubleToLongBits(executeLispExpr("(turbulence x 0.2 0.3 x)", Double.POSITIVE_INFINITY, 0)));

    assertEquals(-1,   executeLispExpr("(add 0 xi)", 1, 0), ScalarVM.epsilon);
    assertEquals(-1,   executeLispExpr("(add 0 yi)", 0, 1), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(add 0 r)", 0, 1), ScalarVM.epsilon);
    assertEquals(Math.sqrt(2),    executeLispExpr("(add 0 r)", 1, 1), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(add 0 r)", 1, 0), ScalarVM.epsilon);
    assertEquals(1,   executeLispExpr("(add 0 ri)", 0, 0), ScalarVM.epsilon);
    assertEquals(0,   executeLispExpr("(add 0 ri)", 1, 0), ScalarVM.epsilon);
    assertEquals(1-Math.sqrt(2),  executeLispExpr("(add 0 ri)", 1, 1), ScalarVM.epsilon);

    assertEquals(0.20674118994423063,  executeLispExpr("(noise 0.1 0.2 0.3)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.18026221874439646,  executeLispExpr("(noise 0.1 0.2 0.1)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.18026221874439646,  executeLispExpr("(noise 0.1 0.2)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.17876534473210404,  executeLispExpr("(noise 0.1 0.1 0.1)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.17876534473210404,  executeLispExpr("(noise 0.1)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.5462180318903666,   executeLispExpr("(turbulence 0.1 0.2 0.3 3000)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.9280194981689174,   executeLispExpr("(turbulence -3000)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.20674118994423063,  executeLispExpr("(snoise 0.1 0.2 0.3 1.0)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.2484739865777904,   executeLispExpr("(snoise 0.1)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.2484739865777904,   executeLispExpr("(sturbulence 0.1)", 1, 0), ScalarVM.epsilon);

    assertEquals(1,    executeLispExpr("(ifPositive 0 1 2 3)", 1, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(ifPositive 0 1 2)", 1, 0), ScalarVM.epsilon);
    assertEquals(-4,   executeLispExpr("(ifPositive (ifPositive -1 2 -3) 1 -4)", 1, 0), ScalarVM.epsilon);
    assertEquals(5,    executeLispExpr("(ifPositive (ifPositive -1 2 -3) 1 (ifPositive 1 5 -3))", 1, 0), ScalarVM.epsilon);
    assertEquals(-10,  executeLispExpr("(ifPositive (ifPositive -1 2 -3) 1 (ifPositive -10 5))", 1, 0), ScalarVM.epsilon);
    assertEquals(-10,  executeLispExpr("(ifPositive (ifPositive -1 2 -3) 1 (ifPositive -10))", 1, 0), ScalarVM.epsilon);

    assertEquals(8,    executeLispExpr("(pow 2 3)", 1, 0), ScalarVM.epsilon);
    assertEquals(0.125,executeLispExpr("(pow 2 -3)", 1, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(log x)", Math.E, 0), ScalarVM.epsilon);
    assertEquals(1.7,  executeLispExpr("(add 0.7 x)", 1, 0), ScalarVM.epsilon);
    assertEquals(10,   executeLispExpr("(add 1 2 x y)", 3, 4), ScalarVM.epsilon);
    assertEquals(120,  executeLispExpr("(mult 1 2 x y 5)", 3, 4), ScalarVM.epsilon);
    assertEquals(120,  executeLispExpr("(mult (add 0 1) (add 1 1) (div x 1 1 2) (mult (add 1 1) y) (sub 15 5 3 2))", 6, 2), ScalarVM.epsilon);
    assertEquals(-0.3, executeLispExpr("(sub 0.7 x)", 1, 0), ScalarVM.epsilon);
    assertEquals(-1,   executeLispExpr("(sub 0.7 x y (div 2 10))", 1, 0.5), ScalarVM.epsilon);
    assertEquals(-1,   executeLispExpr("(sub 0.7 x y (div 2 10))", 1, 0.5), ScalarVM.epsilon);
    assertEquals(0.99, executeLispExpr("(mix x y 0.99)", 0, 1), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(mix -1 1 0.5)", 0, 1), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(step 1 1)", 0, 1), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(step y -0.99)", 0, -1), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(puls -1 1 x)", 0, 1), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(puls 0 0.1 x)", 0-ScalarVM.epsilon, 1), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(puls 0 0.1 x)", 0, 0.1+ScalarVM.epsilon), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(clamp -1 1 x)", 10, 1), ScalarVM.epsilon);
    assertEquals(-1,   executeLispExpr("(clamp -1 1 x)", -(1+ScalarVM.epsilon), 1), ScalarVM.epsilon);
    assertEquals(0.55, executeLispExpr("(clamp 0.5 0.6 x)", 0.55, 1), ScalarVM.epsilon);
    assertEquals(0.1,  executeLispExpr("(min x y)", 0.1, 1.2), ScalarVM.epsilon);
    assertEquals(0.1,  executeLispExpr("(min y x)", 0.1, 1.2), ScalarVM.epsilon);
    assertEquals(0.1,  executeLispExpr("(min y x 0)", 0.1, 1.2), ScalarVM.epsilon);
    assertEquals(-0.1, executeLispExpr("(min y x 0)", -0.1, Double.POSITIVE_INFINITY), ScalarVM.epsilon);
    assertEquals(1.2,  executeLispExpr("(max x y)", 0.1, 1.2), ScalarVM.epsilon);
    assertEquals(-0.1, executeLispExpr("(max y x)", -0.1, -1.2), ScalarVM.epsilon);
    assertEquals(1.2,  executeLispExpr("(max y x 10)", 0.1, 1.2), ScalarVM.epsilon);
    assertEquals(0.7,  executeLispExpr("(abs 0.7)", 1, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(abs x)", -1, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(smoothstep -1 1 x)", 1+ScalarVM.epsilon, 1), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(smoothstep -1 1 x)", -(1+ScalarVM.epsilon), 1), ScalarVM.epsilon);
    assertEquals(0.5,  executeLispExpr("(smoothstep -1 1 x)", 0, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(boxstep -1 1 x)", 1+ScalarVM.epsilon, 1), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(boxstep -1 1 x)", -(1+ScalarVM.epsilon), 1), ScalarVM.epsilon);
    assertEquals(0.5,  executeLispExpr("(boxstep -1 1 x)", 0, 0), ScalarVM.epsilon);
    assertEquals(2,    executeLispExpr("(mod 11 3)", 0, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(mod -11 3)", 0, 0), ScalarVM.epsilon);
    assertEquals(2,    executeLispExpr("(sqrt x)", 4, 0), ScalarVM.epsilon);
    assertEquals(2,    executeLispExpr("(sqrt x)", -4, 0), ScalarVM.epsilon);
    assertEquals(0,    executeLispExpr("(sqrt 0)", 0, 0), ScalarVM.epsilon);
    assertEquals(0.5,  executeLispExpr("(sin x)", Math.PI/6, 0), ScalarVM.epsilon);
    assertEquals(0.5,  executeLispExpr("(cos y)", 0, Math.PI/3), ScalarVM.epsilon);
    assertTrue(executeLispExpr("(tan y)", 0, Math.PI/2) > 1E10);
    assertEquals(-Math.PI/2, executeLispExpr("(atan y)", 0, Double.NEGATIVE_INFINITY), ScalarVM.epsilon);
    assertEquals(Math.PI/4,  executeLispExpr("(atanq x y)", 1, 1), ScalarVM.epsilon);
    assertEquals(-Math.PI/2, executeLispExpr("(atanq x y)", -1, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(floor x)", 1, 0), ScalarVM.epsilon);
    assertEquals(3,    executeLispExpr("(floor x)", Math.PI, 0), ScalarVM.epsilon);
    assertEquals(1,    executeLispExpr("(ceil x)", 1, 0), ScalarVM.epsilon);
    assertEquals(4,    executeLispExpr("(ceil x)", Math.PI, 0), ScalarVM.epsilon);

    assertEquals(38.832975677895, executeLispExpr("(euclide 11 12 x y)", 33, 44), ScalarVM.epsilon);
    assertEquals(54,   executeLispExpr("(absolute 11 12 x y)", 33, 44), ScalarVM.epsilon);

    assertEquals(0.0,  executeLispExpr("(gamma 0.7 x)", 0, 1), ScalarVM.epsilon);
    assertEquals(1.0,  executeLispExpr("(gamma 0.7 y)", 0, 1), ScalarVM.epsilon);
    assertEquals(0.0,  executeLispExpr("(gamma 1.7 x)", 0, 1), ScalarVM.epsilon);
    assertEquals(1.0,  executeLispExpr("(gamma 1.7 y)", 0, 1), ScalarVM.epsilon);
    assertEquals(0.5,  executeLispExpr("(gamma 1.0 0.5)", 0, 1), ScalarVM.epsilon);
    assertEquals(0.0,  executeLispExpr("(bias 0.7 x)", 0, 1), ScalarVM.epsilon);
    assertEquals(1.0,  executeLispExpr("(bias 0.7 y)", 0, 1), ScalarVM.epsilon);
    assertEquals(0.0,  executeLispExpr("(gain 0.7 x)", 0, 1), ScalarVM.epsilon);
    assertEquals(1.0,  executeLispExpr("(gain 0.7 y)", 0, 1), ScalarVM.epsilon);
    assertEquals(0.5,  executeLispExpr("(gain 0.8 0.5)", 0, 1), ScalarVM.epsilon);
    assertEquals(0.0,  executeLispExpr("(gain 1.7 x)", 0, 1), ScalarVM.epsilon);
  }

  public void ignore_testExecute1M() {
    kandid.soup.EntityType entity = (kandid.soup.EntityType) soupFactory.unmarshalFromString(expr1);
    ScalarExprGene scalarExprGene = entity.getScalarExpression().getChromosome().getScalarExpression();
    scalarVM.compile(scalarExprGene);
    for (int i = 0; i < 100; ++i) {
      assertEquals(-0.3, scalarVM.execute(1.0, 2.0, ScalarVM.epsilon), ScalarVM.epsilon);
    }
    long start = System.currentTimeMillis();
    final int rounds = 1000000;
    for (int ix = 0; ix < rounds; ++ix) {
      scalarVM.execute(1.0, 2.0, ScalarVM.epsilon);
    }
    long done = System.currentTimeMillis() - start;
    System.out.println((1000.0 / (double)done) + " mips");
  }

}
