/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.vm.scalar;

import java.awt.Component;
import java.awt.Dimension;

import kandid.calculation.RefinementCalculation;
import kandid.colorator.ColorF32;
import kandid.soup.ChromosomeType;
import kandid.soup.ScalarExprGene;
import kandid.soup.ScalarExpressionChromosome;

public class ScalarExpressionCalculation extends RefinementCalculation {

  private ScanResults scanResults;
  private ScalarVM scalarVM;

  private class ScanResults {
    double  maxValue = Double.NEGATIVE_INFINITY;
    double  minValue = Double.POSITIVE_INFINITY;
    boolean isNaN = false;
    double  range = 0.0001;
    
    public ScanResults(ScalarExpressionChromosome scalarExpressionChromosome) {
      final long nan = Double.doubleToLongBits(Double.NaN);
      double scalarValue = scalarVM.execute(0, 0, ScalarVM.epsilon);
      if (nan == Double.doubleToLongBits(scalarValue)) {
        isNaN = true;
      }
      else {
        maxValue = minValue = scalarValue;
      }
      double xt = xwMin;
      while (xt < xwMax + ScalarVM.epsilon) {
        double yt = ywMin;
        while (yt < ywMax + ScalarVM.epsilon) {
          scalarValue = scalarVM.execute(xt, yt, ScalarVM.epsilon);
          if (nan == Double.doubleToLongBits(scalarValue)) {
            isNaN = true;
          }
          else {
            if (scalarValue < minValue)
              minValue = scalarValue;
            if (scalarValue > maxValue)
              maxValue = scalarValue;
          }
          yt += 0.1;
        }
        xt += 0.1;
      }
      range = maxValue - minValue;
      if (range < 0.0001)
        range = 0.0001;
    }
  
  }

  /* For documentation please see super class RefinementCalculation. */
  public boolean reject(ChromosomeType chromosome) {
    depth = 6;
    xwMin = -1.0;
    xwMax = 1.0;
    ywMin = -1.0;
    ywMax = 1.0;
    this.chromosome = chromosome;
    ScalarExpressionChromosome scalarExpressionChromosome = (ScalarExpressionChromosome) chromosome;
    ScalarExprGene scalarExprGene = scalarExpressionChromosome.getScalarExpression();
    scalarVM = new ScalarVM(scalarExpressionChromosome.getNoiseSeed().getValue());
    try {
      scalarVM.compile(scalarExprGene);
    } catch (Exception e) {
      e.printStackTrace();
      return true;
    }

    scanResults = new ScanResults(scalarExpressionChromosome);
    return scanResults.isNaN || (scanResults.maxValue - scanResults.minValue) < 0.00009;
  }

  /* For documentation please see super class RefinementCalculation. */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    depth = 6;
    xwMin = -1.1;
    xwMax = 1.1;
    ywMin = -1.1;
    ywMax = 1.1;
    super.activateCanvas(viewComponent, canvasSize, zoomMode);

    ScalarExpressionChromosome scalarExpressionChromosome = (ScalarExpressionChromosome) chromosome;
    ScalarExprGene scalarExprGene = scalarExpressionChromosome.getScalarExpression();
    scalarVM = new ScalarVM(scalarExpressionChromosome.getNoiseSeed().getValue());
    scalarVM.compile(scalarExprGene);

    scanResults = new ScanResults((ScalarExpressionChromosome) chromosome);
  }

  /* (non-Javadoc)
   * @see kandid.calculation.Calculation#calculate(boolean)
   */
  public void calculate(boolean paintOnScreen, String doExport) {
    if (enterDepth()) {
      ColorF32 cout = new ColorF32();
      while (more) {
        double scalarValue = scalarVM.execute(xw, yw, ScalarVM.epsilon);
        colorator.getColor((scalarValue - scanResults.minValue) / scanResults.range, cout);
        setPixel(cout);
        nextLocation();
      }
    }
  }

  /* (non-Javadoc)
   * @see kandid.calculation.Calculation#getBackgroundColor()
   */
  public boolean hasWhiteBackground() {
    return false;
  }

}
