/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.symmifs;

import kandid.calculation.*;
import kandid.colorator.BlackWhiteColorator;
import kandid.colorator.Colorator;
import kandid.colorator.GrayColorator;
import kandid.soup.ChromosomeType;
import kandid.soup.SymmetricIfsChromosome;
import kandid.util.*;

/**
 * Class SymIFSCalculation
 * @author thomas jourdan
 */
public class SymmetricIfsCalculation extends IterationCalculation implements Cloneable {
  int    symmetry;
  int    p;
  double lambda;
  double alpha;
  double beta;
  double gamma;
  double omega;

  /**
   * Number of iterations depends on coloring type.
   * @param width
   * @param height
   * @return int maximum number of iterations
   */
  protected int getMaxIterations(int width, int height) {
    if (colorator instanceof BlackWhiteColorator) {      
      return 4 * 8 * canvasSize.width * canvasSize.height;
    }
    if (colorator instanceof GrayColorator) {      
      return 4 * 16 * canvasSize.width * canvasSize.height;
    }
    return 4 * 64 * canvasSize.width * canvasSize.height;
  }
  
  
  /**
   * map chromosome to local member variables
   * @param chromosome
   */
  public void mapChromosome(ChromosomeType chromosome) {
    SymmetricIfsChromosome symmetricIfsChromosome = (SymmetricIfsChromosome)chromosome;
    symmetry = symmetricIfsChromosome.getSymmetry().getValue();
    p        = -1;
    lambda   = symmetricIfsChromosome.getLambda().getValue();
    alpha    = symmetricIfsChromosome.getAlpha().getValue();
    beta     = symmetricIfsChromosome.getBeta().getValue();
    gamma    = symmetricIfsChromosome.getGamma().getValue();
    omega    = symmetricIfsChromosome.getOmega().getValue();
  }

  /**
   * @see kandid.calculation.Calculation#setChromosome(ChromosomeType, Colorator)
   */
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
    mapChromosome(chromosome);
  }

  /**
   * Tests if this chromosome is able to produce a picture.
   *
   * @return true if the calculation does not accept the chromosome.
   */
  public boolean reject(ChromosomeType chromosome) {
    mapChromosome(chromosome);
    startIteration();
    findExtremeValues(500);
    if (Debug.enabled) System.out.println("xmin " + xmin + " xmin " + xmin + " ymin " + ymin + " ymax " + ymax);
    return xmin < -1.0 || xmin > 1.0 || ymin < -1.0 || ymax > 1.0;
  }
  
  /**
   * Calculates the next step for this iteration.
   * Results can be fond in member variable yw and xw.
   */
  protected void iterate() {
    if (p >= 0) {
      int     ix;
      Complex z     = new Complex(xw, yw);
      Complex term1 = z;
      for (ix = 1; ix < symmetry; ++ix) {
        term1 = Complex.multiply(term1, z);
      }

      Complex term2;
      if (p >= 0) {
        Complex factor = Complex.divide(z, Complex.abs(z));
        int     power   = p * symmetry;
        if (power > 0) {
          term2 = factor;
          for (ix = 1; ix < power; ++ix) {
            term2 = Complex.multiply(term2, factor);
          }
        }
        else {
          term2 = new Complex(1.0, 0.0);
        }
      }
      else {
        term2 = new Complex(1.0, 0.0);
      }
      Complex term3 = Complex.conj(z);
      for (ix = 2; ix < symmetry; ++ix) {
        term3 = Complex.multiply(term3, Complex.conj(z));
      }
      //!!    znew = (lambda+alpha*z*conj(z)+beta*term1+omega*real(term2)*abs(z))*z+gamma*term3;
      Complex term4 = Complex.multiply(alpha, z);
      term4 = Complex.multiply(term4, Complex.conj(z));
      Complex term5 = Complex.multiply(beta, term1);
      double  term6 = omega * Complex.real(term2) * Complex.abs(z);
      Complex znew  = Complex.add(lambda, term4);
      znew = Complex.add(znew, term5);
      znew = Complex.add(znew, term6);
      znew = Complex.multiply(znew, z);
      znew = Complex.add(znew, Complex.multiply(gamma, term3));
      xw   = Complex.real(znew);
      yw   = Complex.imag(znew);
    }
    else {
      double x2, y2;
      double zbar  = xw * xw + yw * yw;
      double zReal = xw;
      double zImag = yw;
      for (int ix = 0; ix < symmetry - 2; ++ix) {
        double zR = zReal * xw - zImag * yw;
        double zI = zImag * xw + zReal * yw;
        zReal = zR;
        zImag = zI;
      }
      double zPowerN = xw * zReal - yw * zImag;
      double p       = lambda + alpha * zbar + beta * zPowerN;
      x2 = p * xw + gamma * zReal - omega * yw;
      y2 = p * yw - gamma * zImag + omega * xw;
      xw = x2;
      yw = y2;
    }
  }
  
  /**
   * white background.
   * @return 0xffffff
   */
  public boolean hasWhiteBackground() {
    return true;
  }

}

