/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.lsys;

import javax.xml.bind.JAXBElement;

import junit.framework.TestCase;
import kandid.soup.LsysILChromosome;
import kandid.soup.util.SoupFactory;

public class LsysILCalculationTest extends TestCase {

  static final String header =
    "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>"
      + "<entity version=\"0.1.3\">"
      + "    <lsysIL version=\"0.2.7\" rating=\"abort\">"
      + "        <calculationName>kandid.calculation.lsys.LsysCalculation</calculationName>"
      + "        <coloratorName>kandid.colorator.BlackWhiteColorator</coloratorName>"
      + "        <chromosome>"
      + "            <ident>LsysIL_a_2</ident>";

  static final String footer = 
    "        </chromosome>"
     + "    </lsysIL>"
      + "</entity>";

  private SoupFactory soupFactory;
  /**
   * Constructor for LsysILCalculationTest.
   * @param name
   */
  public LsysILCalculationTest(String name) {
    super(name);
    soupFactory = SoupFactory.getSoupFactory();
  }

  private static String buildProduction(String axiom, String[] predecessor, String[] successor, String[] leftContext, String[] rightContext, int depth) {
    String ret;
      ret =   header
        + "            <alphabet>"
        + "                <value>FAB-+</value>"
        + "            </alphabet>"
        + "            <axiom>"
        + "                <value>"  + axiom      + "</value>"
        + "            </axiom>";
      assert predecessor.length == successor.length;
      for (int ip = 0; ip < predecessor.length; ip++) {
        ret += "            <production>"
          + "                <predecessor>"
          + "                    <value>" + predecessor[ip] + "</value>"
          + "                </predecessor>"
        + "                <successor>"
        + "                    <value>" + successor[ip] + "</value>"
        + "                </successor>"
        + "                <leftContext>"
        + "                    <value>" + leftContext[ip] + "</value>"
        + "                </leftContext>"
        + "                <rightContext>"
        + "                    <value>" + rightContext[ip] + "</value>"
        + "                </rightContext>"
          + "            </production>";
      }
      ret += 
          "            <directions>"
        + "                <value>4</value>"
        + "            </directions>"
        + "            <baseIndex>"
        + "                <value>10</value>"
        + "            </baseIndex>"
        + "            <ignore>"
        + "                <value></value>"
        + "            </ignore>"
        + "            <depth>"
        + "                <value>"      + depth      + "</value>"
        + "            </depth>"
        + "            <jitter>"
        + "                <value>10</value>"
        + "            </jitter>"
        + "            <noiseSeed>"
        + "                <value>1512</value>"
        + "            </noiseSeed>"
          + footer;
    return ret;
  }
  
  private LsysILChromosome buildChromosome(String axiom, String predecessor[], String successor[], String[] leftContext, String[] rightContext, int depth) {
    String production = buildProduction(axiom, predecessor, successor, leftContext, rightContext, depth);
    javax.xml.bind.JAXBElement uo = (JAXBElement) soupFactory.unmarshalFromString(production);
    kandid.soup.EntityType entity = (kandid.soup.EntityType) uo.getValue();
    LsysILChromosome lsysILChromosome = entity.getLsysIL().getChromosome();
    return lsysILChromosome;
  }

  private String executeProduction(String axiom, String predecessor[], String successor[], String[] leftContext, String[] rightContext, int depth) {
    System.out.println("----\n");
    LsysILChromosome lsysILChromosome = buildChromosome(axiom, predecessor, successor, leftContext, rightContext, depth);
    LsysMashine.testInitCalculation(lsysILChromosome);
    LsysMashine.expand(lsysILChromosome.getDepth().getValue());
    byte[] result = LsysMashine.string;
    StringBuffer ret = new StringBuffer(LsysMashine.length);
    for (int rx = 0; rx < LsysMashine.length; rx++) {
      ret.append((char)result[rx]);
    }
    System.out.println(ret.toString());
    return ret.toString();
  }

  public void testExpand() {
    {
      String axiom =            "F";
      String predecessor[] =  { "F", "F" };
      String leftContext[] =  { "",  "" };
      String rightContext[] = { "",  "" };
      String successor[] =    { "-", "FF-[-F+F+F]+[+F-F-F]"};
      assertEquals(
       "FF-[-F+F+F]+[+F-F-F]--[-FF-[-F+F+F]+[+F-F-F]+-+FF-[-F+F+F]+[+F-F-F]]+[+--FF-[-F+F+F]+[+F-F-F]-FF-[-F+F+F]+[+F-F-F]]",
       executeProduction(axiom, predecessor, successor, leftContext, rightContext, 2));
    }
    
    {
      String axiom =            "BAAAAAAAA";
      String predecessor[] =  { "A", "B" };
      String leftContext[] =  { "B", "" };
      String rightContext[] = { "",  "" };
      String successor[] =    { "B", "A"};
      assertEquals(
       "ABAAAAAAA",
       executeProduction(axiom, predecessor, successor, leftContext, rightContext, 1));
      assertEquals(
       "AAAAAAAAB",
       executeProduction(axiom, predecessor, successor, leftContext, rightContext, 8));
    }
    
  }

}
