/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.lsys;

import javax.xml.bind.JAXBElement;

import junit.framework.TestCase;
import kandid.soup.LsysD0LChromosome;
import kandid.soup.genetic.lsys.LsysGenetic;
import kandid.soup.util.SoupFactory;

public class LsysD0LCalculationTest extends TestCase {

  static final String header =
    "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>"
      + "<entity version=\"0.1.3\">"
      + "    <lsysD0L version=\"0.2.7\" rating=\"abort\">"
      + "        <calculationName>kandid.calculation.lsys.LsysCalculation</calculationName>"
      + "        <coloratorName>kandid.colorator.BlackWhiteColorator</coloratorName>"
      + "        <chromosome>"
      + "            <ident>Lsys_dqhq698a_dqhq6992</ident>";

  static final String footer = 
    "        </chromosome>"
     + "    </lsysD0L>"
      + "</entity>";

  private SoupFactory soupFactory;
  /**
   * Constructor for LsysD0LCalculationTest.
   * @param name
   */
  public LsysD0LCalculationTest(String name) {
    super(name);
    soupFactory = SoupFactory.getSoupFactory();
  }

  public static String buildProduction(String axiom, String[] predecessor, String[] successor, int depth) {
    String ret;
      ret =   header
        + "            <alphabet>"
        + "                <value>F-+-</value>"
        + "            </alphabet>"
        + "            <axiom>"
        + "                <value>"  + axiom      + "</value>"
        + "            </axiom>";
      assert predecessor.length == successor.length;
      for (int ip = 0; ip < predecessor.length; ip++) {
        ret += "            <production>"
          + "                <predecessor>"
          + "                    <value>" + predecessor[ip] + "</value>"
          + "                </predecessor>"
          + "                <successor>"
          + "                    <value>" + successor[ip] + "</value>"
          + "                </successor>"
          + "            </production>";
      }
      ret += 
          "            <directions>"
        + "                <value>4</value>"
        + "            </directions>"
        + "            <baseIndex>"
        + "                <value>10</value>"
        + "            </baseIndex>"
        + "            <depth>"
        + "                <value>"      + depth      + "</value>"
        + "            </depth>"
          + footer;
    return ret;
  }
  
  private LsysD0LChromosome buildChromosome(String axiom, String predecessor[], String successor[], int depth) {
    String production = buildProduction(axiom, predecessor, successor, depth);
    javax.xml.bind.JAXBElement uo = (JAXBElement) soupFactory.unmarshalFromString(production);
    kandid.soup.EntityType entity = (kandid.soup.EntityType) uo.getValue();
    LsysD0LChromosome lsysD0LChromosome = entity.getLsysD0L().getChromosome();
    return lsysD0LChromosome;
  }

  private String executeProduction(String axiom, String predecessor[], String successor[], int depth) {
    System.out.println("----\n");
    LsysD0LChromosome lsysD0LChromosome = buildChromosome(axiom, predecessor, successor, depth);
    LsysMashine.testInitCalculation(lsysD0LChromosome);
    LsysMashine.expand(lsysD0LChromosome.getDepth().getValue());
    byte[] result = LsysMashine.string;
    StringBuffer ret = new StringBuffer(LsysMashine.length);
    for (int rx = 0; rx < LsysMashine.length; rx++) {
      ret.append((char)result[rx]);
    }
    System.out.println(ret.toString());
    return ret.toString();
  }

  public void testErzeugePflanze() {
    {
      String predecessor[] = { "F" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]" };
      assertEquals("F", executeProduction("F", predecessor, successor, 0));
      assertEquals("FF-[-F+F+F]+[+F-F-F]", executeProduction("F", predecessor, successor, 1));
      assertEquals(
        "FF-[-F+F+F]+[+F-F-F]FF-[-F+F+F]+[+F-F-F]-[-FF-[-F+F+F]+[+F-F-F]+FF-[-F+F+F]+[+F-F-F]+FF-[-F+F+F]+[+F-F-F]]+[+FF-[-F+F+F]+[+F-F-F]-FF-[-F+F+F]+[+F-F-F]-FF-[-F+F+F]+[+F-F-F]]",
        executeProduction("F", predecessor, successor, 2));
    }

    {
      String predecessor[] = { "F" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]" };
      assertFalse(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 0)));
    }
    
  }

  public void testReject() {
    {
      String predecessor[] = { "F" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]" };
      assertFalse(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 1)));
    }
    
    {
      String predecessor[] = { "F", "+" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]", "-" };
      assertFalse(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 1)));
    }
    
    {
      String predecessor[] = { "F", "F" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]", "-" };
      assertTrue(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 1)));
    }
    
  }

}
