/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.lsys;

import javax.xml.bind.JAXBElement;

import junit.framework.TestCase;
import kandid.soup.Lsys0LChromosome;
import kandid.soup.genetic.lsys.LsysGenetic;
import kandid.soup.util.SoupFactory;

public class Lsys0LCalculationTest extends TestCase {

  static final String header =
    "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>"
      + "<entity version=\"0.1.3\">"
      + "    <lsys0L version=\"0.2.7\" rating=\"abort\">"
      + "        <calculationName>kandid.calculation.lsys.LsysCalculation</calculationName>"
      + "        <coloratorName>kandid.colorator.BlackWhiteColorator</coloratorName>"
      + "        <chromosome>"
      + "            <ident>Lsys_dqhq698a_dqhq6992</ident>";

  static final String footer = 
    "        </chromosome>"
     + "    </lsys0L>"
      + "</entity>";

  private SoupFactory soupFactory;
  /**
   * Constructor for Lsys0LCalculationTest.
   * @param name
   */
  public Lsys0LCalculationTest(String name) {
    super(name);
    soupFactory = SoupFactory.getSoupFactory();
  }

  private static String buildProduction(String axiom, String[] predecessor, String[] successor, int depth) {
    String ret;
      ret =   header
        + "            <alphabet>"
        + "                <value>F-+-</value>"
        + "            </alphabet>"
        + "            <axiom>"
        + "                <value>"  + axiom      + "</value>"
        + "            </axiom>";
      assert predecessor.length == successor.length;
      for (int ip = 0; ip < predecessor.length; ip++) {
        ret += "            <production>"
          + "                <predecessor>"
          + "                    <value>" + predecessor[ip] + "</value>"
          + "                </predecessor>"
          + "                <successor>"
          + "                    <value>" + successor[ip] + "</value>"
          + "                </successor>"
          + "            </production>";
      }
      ret += 
          "            <directions>"
        + "                <value>4</value>"
        + "            </directions>"
        + "            <baseIndex>"
        + "                <value>10</value>"
        + "            </baseIndex>"
        + "            <depth>"
        + "                <value>"      + depth      + "</value>"
        + "            </depth>"
        + "            <jitter>"
        + "                <value>10</value>"
        + "            </jitter>"
        + "            <noiseSeed>"
        + "                <value>1512</value>"
        + "            </noiseSeed>"
          + footer;
    return ret;
  }
  
  private Lsys0LChromosome buildChromosome(String axiom, String predecessor[], String successor[], int depth) {
    String production = buildProduction(axiom, predecessor, successor, depth);
    javax.xml.bind.JAXBElement uo = (JAXBElement) soupFactory.unmarshalFromString(production);
    kandid.soup.EntityType entity = (kandid.soup.EntityType) uo.getValue();
    Lsys0LChromosome lsys0LChromosome = entity.getLsys0L().getChromosome();
    return lsys0LChromosome;
  }

  private String executeProduction(String axiom, String predecessor[], String successor[], int depth) {
    System.out.println("----\n");
    Lsys0LChromosome lsys0LChromosome = buildChromosome(axiom, predecessor, successor, depth);
    LsysMashine.testInitCalculation(lsys0LChromosome);
    LsysMashine.expand(lsys0LChromosome.getDepth().getValue());
    byte[] result = LsysMashine.string;
    StringBuffer ret = new StringBuffer(LsysMashine.length);
    for (int rx = 0; rx < LsysMashine.length; rx++) {
      ret.append((char)result[rx]);
    }
    System.out.println(ret.toString());
    return ret.toString();
  }

  public void testErzeugePflanze() {
    {
      String predecessor[] = { "F", "F" };
      String successor[] = { "-", "FF-[-F+F+F]+[+F-F-F]"};
      assertEquals(
       "FF-[-F+F+F]+[+F-F-F]--[-FF-[-F+F+F]+[+F-F-F]+-+FF-[-F+F+F]+[+F-F-F]]+[+--FF-[-F+F+F]+[+F-F-F]-FF-[-F+F+F]+[+F-F-F]]",
       executeProduction("F", predecessor, successor, 2));
    }
    
  }

  public void testReject() {
    {
      String predecessor[] = { "F" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]" };
      assertFalse(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 1)));
    }
    
    {
      String predecessor[] = { "F", "+" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]", "-" };
      assertFalse(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 1)));
    }
    
    {
      String predecessor[] = { "F", "F" };
      String successor[] = { "FF-[-F+F+F]+[+F-F-F]", "-" };
      assertFalse(LsysGenetic.reject(buildChromosome("F", predecessor, successor, 1)));
    }
    
  }
  
}
