/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.bridge;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.IOException;

import junit.framework.TestCase;
import kandid.calculation.bridge.povray.PovJob;
import kandid.soup.ChromosomeType;
import kandid.soup.PopulationType;
import kandid.soup.test.SoupFactoryTest;
import kandid.soup.util.SoupFactory;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 *
 */
public class StreamBridgeTest extends TestCase {

  private static final int height = 25;
  private static final int width = 25;
  SoupFactory soupFactory;
  PopulationType population;
  ChromosomeType aChromosome;

  Dimension dim = new Dimension(width, height);

  /**
   * Constructor for StreamBridgeTest.
   * @param name
   */
  public StreamBridgeTest(String name) {
    super(name);
  }

  /* (non-Javadoc)
   * @see junit.framework.TestCase#setUp()
   */
  protected void setUp() throws Exception {
    super.setUp();
    soupFactory = SoupFactory.getSoupFactory();
    population = SoupFactoryTest.createRandomPopulation(soupFactory, "PovIsoSurface", "PovTextureCalculation", "kandid.colorator.RGBColorator");
    aChromosome = soupFactory.getChromosome(population, 0);

    dim = new Dimension(width, height);
  }

  public void testExecute() {
    PovJob povJob = new PovJob();
    povJob.createJob(aChromosome, dim, false, null, "png");
    StreamBridge streamBridge = new StreamBridge(povJob);
    streamBridge.execute();
    BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
    boolean ready = false;
    do {
      ready = streamBridge.recieve(image.getGraphics());
    }
    while (!ready);
    assertNotNull(image);
    assertEquals(height, image.getHeight());
    assertEquals(width, image.getWidth());
    assertEquals(BatchJob.stateReady, povJob.state);
    streamBridge.waitFor();
    streamBridge.cleanUp();
  }

  public void testDestroy() {
    PovJob povJob = new PovJob();
    povJob.createJob(aChromosome, dim, false, null, "png");
    StreamBridge streamBridge = new StreamBridge(povJob);
    streamBridge.execute();
    streamBridge.destroy();
    assertEquals(BatchJob.stateDestroied, povJob.state);
    BufferedImage image = null;
    try {
      image = streamBridge.getImage();
    }
    catch (IOException exc) {
      Debug.stackTrace(exc);
      fail();
    }
    assertNull(image);
  }

  public void testErrorMessage() {
    PovJob povJob = new PovJob();
    povJob.createJob(aChromosome, dim, false, null, "png");
    povJob.script = povJob.script.replace('<', '!');
    StreamBridge streamBridge = new StreamBridge(povJob);
    streamBridge.execute();
    streamBridge.waitFor();
    assertTrue(povJob.errorMessage.length() >= 0);
    //System.out.println(povJob.errorMessage.toString());
    assertTrue(povJob.errorMessage.toString().indexOf("Fatal error in parser:") > 0);
  }

}
