/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.bridge;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.IOException;

import kandid.Kandid;
import kandid.soup.ChromosomeType;
import kandid.util.Debug;

//TODO clone population and cache
//TODO clone image and cache

/**
 * @author thomas jourdan
 *
 */
public abstract class BatchJob {
  public static final int statePending = 0;
  public static final int stateRunning = 1;
  public static final int stateReady = 2;
  public static final int stateDestroied = 3;
  public int state = statePending;

  public String childProgram;
  public String[] comandline;
  public String[] environment;
  public String script;
  public String scriptFileName;
  public String imageFileName;
  public String imageFormat;
  public boolean doExport;
  public Dimension canvasSize;
  public boolean preview;
  public Object sync;
  public int exitCode;
  public int expextedExitCode;
  public StringBuffer errorMessage = new StringBuffer();

  public Object incrementReady = new Object();
  public BufferedImage importedImage;

  protected static boolean firstUsage = true;
  protected static int showExecError = 0;

  /**
   * @return
   */
  protected abstract String[] buildCommand();

  /**
   * @return
   */
  protected abstract String[] buildEnvironment();

  /**
   * @param chromosome
   * @return
   */
  protected abstract String expandScript(ChromosomeType chromosome);

  /**
   * @return
   */
  protected abstract String getProgramName();

  /**
   * @return
   */
  protected abstract String getScriptExtension();

  /**
   * 
   */
  protected abstract void showExecError();

  /**
   * @param chromosome
   * @param canvasSize
   * @param preview
   * @param exportFilename
   * @param imageFormat
   */
  public void createJob(
    ChromosomeType chromosome,
    Dimension canvasSize,
    boolean preview,
    String exportFilename,
    String imageFormat) {
    this.canvasSize = canvasSize;
    this.preview = preview;
    this.imageFormat = imageFormat;
    doExport = exportFilename != null;
    expextedExitCode = 1;
    errorMessage = new StringBuffer();
    childProgram = getProgramName();
    String ident = chromosome.getIdent() + (preview ? "__p" : "__f");
    if (doExport) {
      scriptFileName = exportFilename + getScriptExtension();
      imageFileName = exportFilename + "." + imageFormat;
    }
    else {
      scriptFileName = Kandid.scratchFolder + "/" + ident + getScriptExtension();
      imageFileName = Kandid.scratchFolder + "/" + ident + "." + imageFormat;
    }
    script = expandScript(chromosome);
    comandline = buildCommand();
    environment = buildEnvironment();

    if (firstUsage) {
      firstUsage = false;
      tryExecute();
    }
  }

  /**
   * @return
   */
  public boolean tryExecute() {
    Process childProcess = null;
    boolean found = false;
    try {
      String[] cmdarray = new String[1];
      cmdarray[0] = getProgramName();
      childProcess = Runtime.getRuntime().exec(cmdarray);
      found = true;
    }
    catch (IOException exc) {
      // child process can not be started
      Debug.stackTrace(exc);
      showExecError();
    }
    if (childProcess != null)
      childProcess.destroy();
    return found;
  }
}
