/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation.affineifs;

import java.util.*;
import kandid.calculation.*;
import kandid.colorator.*;
import kandid.soup.AffineIfsChromosome;
import kandid.soup.AffineTransformationGene;
import kandid.soup.ChromosomeType;
import kandid.util.*;

/**
 * Class AffineIfsCalculation
 * @author thomas jourdan
 */
public class AffineIfsCalculation extends IterationCalculation implements Cloneable {
  protected Random         rand            = new Random();
//!!  protected int            maxTransformations = AffineIfsChromosome.getAffineTransformation();
  protected int            maxTransformations = 8;
  protected int            numTransformations;
  private double           mTa[]           = new double[maxTransformations];
  private double           mTb[]           = new double[maxTransformations];
  private double           mTc[]           = new double[maxTransformations];
  private double           mTd[]           = new double[maxTransformations];
  private double           mTe[]           = new double[maxTransformations];
  private double           mTf[]           = new double[maxTransformations];
  private static final int maxMatcher         = 128;
  private int              matcher[] = new int[maxMatcher];
  
  
  /**
   * Number of iterations depends on coloring type.
   * @param width
   * @param height
   * @return int maximum number of iterations
   */
  protected int getMaxIterations(int width, int height) {
    if (colorator instanceof BlackWhiteColorator) {      
      return 8 * canvasSize.width * canvasSize.height;
    }
    if (colorator instanceof GrayColorator) {      
      return 16 * canvasSize.width * canvasSize.height;
    }
    return 64 * canvasSize.width * canvasSize.height;
  }
  
  
  /**
   * Tests if this chromosome is able to produce a picture.
   *
   * @return true if the calculation does not accept the chromosome.
   */
  public boolean reject(ChromosomeType chromosome) {
    this.chromosome     = chromosome;
    prepareMatcher((AffineIfsChromosome)chromosome);
    startIteration();
    findExtremeValues(5000);
    if (Debug.enabled) System.out.println("xmin " + xmin + " xmin " + xmin + " ymin " + ymin + " ymax " + ymax);
    return xmin < -1000.0 || xmax > 1000.0 || ymin < -1000.0 || ymax > 1000.0;
  }
  
  /* For documentation please see super class Calculation. */
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    super.setChromosome(chromosome, colorator);
    prepareMatcher((AffineIfsChromosome)chromosome);
  }

  /**
   * Method prepareMatcher
   *
   * @param ifsChromosome
   */
  protected void prepareMatcher(AffineIfsChromosome ifsChromosome) {
    java.util.List<AffineTransformationGene> transformationList = ifsChromosome.getAffineTransformation();
    numTransformations = transformationList.size();
    /** array of probabilities summing to 1 */
    double P[] = new double[maxTransformations];
    rand.setSeed(ifsChromosome.getSeed().getValue());
    double sum = 0.0;
    // calculate probability for each transformation
    int ax = 0;
    while (ax < numTransformations) {
      AffineTransformationGene afGene = transformationList.get(ax);
      double result[] = Util.polar2matrix(afGene.getTa().getValue(),
                                          afGene.getTb().getValue(),
                                          afGene.getTc().getValue(),
                                          afGene.getTd().getValue(),
                                          afGene.getTe().getValue(),
                                          afGene.getTf().getValue());
      mTa[ax] = result[0];
      mTb[ax] = result[1];
      mTc[ax] = result[2];
      mTd[ax] = result[3];
      mTe[ax] = result[4];
      mTf[ax] = result[5];
      ++ax;
    }
    for (ax = 0; ax < maxTransformations; ++ax) {
      P[ax] = 0.0;
    }

    ax = 0;
    while (ax < numTransformations) {
      P[ax] = Math.abs(mTa[ax] * mTd[ax] - mTb[ax] * mTc[ax]);
      if (P[ax] < 0.01) {
        P[ax] = 0.01;
      }
      sum += P[ax];
      ++ax;
    }
    P[0] /= sum;
    P[1] /= sum;
    P[2] /= sum;
    P[3] /= sum;
    P[4] /= sum;
    P[5] /= sum;
    P[6] /= sum;
    P[7] /= sum;
    double trigger = 0.0;
    for (int cx = 0; cx < maxMatcher; cx++) {
      if (trigger < P[0]) {
        matcher[cx] = 0;
      }
      else if (trigger < P[0] + P[1]) {
        matcher[cx] = 1;
      }
      else if (trigger < P[0] + P[1] + P[2]) {
        matcher[cx] = 2;
      }
      else if (trigger < P[0] + P[1] + P[2] + P[3]) {
        matcher[cx] = 3;
      }
      else if (trigger < P[0] + P[1] + P[2] + P[3] + P[4]) {
        matcher[cx] = 4;
      }
      else if (trigger < P[0] + P[1] + P[2] + P[3] + P[4] + P[5]) {
        matcher[cx] = 5;
      }
      else if (trigger < P[0] + P[1] + P[2] + P[3] + P[4] + P[5] + P[6]) {
        matcher[cx] = 6;
      }
      else {
        matcher[cx] = 7;
      }
      trigger += 1.0 / (double)maxMatcher;
    }
  }

  /**
   * Calculates the next step for this iteration.
   * Results can be fond in member variable yw and xw.
   */
  protected void iterate() {
    int tx;
    if (numTransformations > 1) {
      tx = matcher[rand.nextInt(maxMatcher)];
    }
    else {
      tx = 0;
    }
    double x2 = xw * mTa[tx] + yw * mTb[tx] + mTe[tx];
    double y2 = xw * mTc[tx] + yw * mTd[tx] + mTf[tx];
    xw = x2;
    yw = y2;
  }
  
  /**
   * white background.
   * @return 0xffffff
   */
  public boolean hasWhiteBackground() {
    return true;
  }

}

