/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation;

import java.awt.*;

import kandid.util.*;

/**
 * Base class for vector graphics.
 * @author thomas jourdan
 */
public abstract class VectorCalculation extends Calculation {
  public Graphics2D g2d;
  public double xmin;
  public double xmax;
  public double ymin;
  public double ymax;
  protected double xfactor;
  protected double yfactor;
  public int xp;
  public int yp;

  /**
   * activate painting for vector based calculation
   *
   * @param canvasSize width and height of the painting canvas.
   * @param g2d Graphics context for paint canvas.
   */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, Graphics2D g2d, boolean zoomMode) {
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    this.viewComponent = viewComponent;
    if (Debug.enabled)
      assert g2d != null : "Graphics2D is undefined";
    this.g2d = g2d;
    xmin = ymin = 0.0;
    xmax = ymax = 1.0;
    xfactor = (double) canvasSize.width;
    yfactor = (double) canvasSize.height;
    xp = yp = 0;
  }

  /**
   * Initialize transformation from word coordinates to pixel coordinates.
   *
   * @param borderFactor factor for the size of a surrounding area.
   */
  public void initTransformation(double borderFactor) {
    double xBorder = borderFactor * (xmax - xmin);
    double yBorder = borderFactor * (ymax - ymin);
    xmax += xBorder;
    xmin -= xBorder;
    ymax += yBorder;
    ymin -= yBorder;
    double diffX = xmax - xmin;
    double diffY = ymax - ymin;
    if (diffX > 0.0) {
      xfactor = (double) canvasSize.width / diffX;
    }
    if (diffY > 0.0) {
      yfactor = (double) canvasSize.height / diffY;
    }
  }

  /**
   * Transform from world coordinate system to pixel coordinate system.
   */
  public void transform(double xw, double yw) {
    xp = (int) ((xw - xmin) * xfactor);
    yp = (int) ((yw - ymin) * yfactor);
  }

}
