/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.Iterator;

import javax.imageio.ImageWriter;
import javax.imageio.stream.FileImageOutputStream;

import ar.com.hjg.pngj.ImageInfo;
import ar.com.hjg.pngj.ImageLineInt;
import ar.com.hjg.pngj.PngWriter;
import kandid.colorator.ColorF32;
import kandid.extensions.ExportState;
import kandid.util.Debug;

/**
 * Base class for pixel based graphics.
 * 
 * @author thomas jourdan
 */
public abstract class PixelCalculation extends Calculation implements Cloneable {
  protected int size;
  protected BufferedImage bufAwtImg;
  protected ColorF32[][] bufColorF32Img;
  protected AffineTransform affineTransform;

  /**
   * activate painting for pixel based calculation
   *
   * @param canvasSize width and height of the painting canvas.
   * @param g2d        Graphics context for paint canvas.
   */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    super.activateCanvas(viewComponent, canvasSize, zoomMode);
    affineTransform = new AffineTransform();
    affineTransform.setToIdentity();
    bufAwtImg = new BufferedImage(canvasSize.width, canvasSize.height, BufferedImage.TYPE_INT_RGB);

    bufColorF32Img = new ColorF32[canvasSize.height][];
    for (int iy = 0; iy < canvasSize.height; ++iy) {
      bufColorF32Img[iy] = new ColorF32[canvasSize.width];
      for (int ix = 0; ix < canvasSize.width; ++ix) {
        bufColorF32Img[iy][ix] = new ColorF32();
      }
    }
  }

  /**
   * Returns the buffered image.
   *
   * @return the buffered image.
   */
  public BufferedImage getImage() {
    return bufAwtImg;
  }

  /**
   * Sets a pixel in to the buffered image.
   *
   * @param xp      x position.
   * @param yp      y position.
   * @param palette 24 Bit color value
   */
  protected void setPixel(int xp, int yp, ColorF32 colorF32) {
    try {
      bufAwtImg.setRGB(xp, yp, ColorF32.toRGB8(colorF32));
      ColorF32.copy(colorF32, bufColorF32Img[yp][xp]);
    } catch (ArrayIndexOutOfBoundsException exc) {
      if (Debug.enabled)
        System.out.println("bufImg coordinates out of bounds (" + xp + ", " + yp + ")");
    }
  }

  /**
   * Method getAffineTransform.
   * 
   * @return AffineTransform
   */
  public AffineTransform getAffineTransform() {
    return affineTransform;
  }

  /**
   * Write 16 Bit PNG File
   * 
   * @param exportState
   * @param imageOutFile
   */
  public void writeRGB_PNG16(ExportState exportState, File imageOutFile) {
    ImageInfo image_info = new ImageInfo(exportState.xDim, exportState.yDim, 16, false);
    PngWriter png_writer = new PngWriter(imageOutFile, image_info);
    ImageLineInt line = new ImageLineInt(image_info);
    int[] scanline = line.getScanline();
    for (int yp = 0; yp < exportState.yDim; ++yp) {
      int j = 0;
      for (int xp = 0; xp < exportState.xDim; ++xp) {
        scanline[j++] = (int) (bufColorF32Img[yp][xp].r * 65535);
        scanline[j++] = (int) (bufColorF32Img[yp][xp].g * 65535);
        scanline[j++] = (int) (bufColorF32Img[yp][xp].b * 65535);
      }
      png_writer.writeRow(line, yp);
    }
    png_writer.end();
  }

  /**
   * Write Portable FloatMap Image Format PFM File
   * see: https://www.pauldebevec.com/Research/HDR/PFM/
   * see: https://www.nayuki.io/page/portable-floatmap-format-io-java
   * original java code by Nayuki, (MIT License)
   * 
   * @param exportState
   * @param imageOutFile
   */
  public void writeRGB_PFM(ExportState exportState, File imageOutFile) {
    // Write header text data. Must use Unix newlines, not universal style
    FileOutputStream out = null;
    PrintWriter header_out = null;
    DataOutput data_out = null;
    try {
      out = new FileOutputStream(imageOutFile);
      header_out = new PrintWriter(new OutputStreamWriter(out, StandardCharsets.US_ASCII));
      header_out.print("PF\n");
      header_out.print(exportState.xDim + " " + exportState.yDim + "\n");
      header_out.print("-1.0\n");
      header_out.flush();
      // Detach the PrintWriter stream

      // Write float32 image pixel data
      data_out = new DataOutputStream(out);
      for (int yp = exportState.yDim-1; yp >= 0; --yp) {
        for (int xp = 0; xp < exportState.xDim; ++xp) {
          data_out.writeInt(Integer.reverseBytes(Float.floatToIntBits(bufColorF32Img[yp][xp].r)));
          data_out.writeInt(Integer.reverseBytes(Float.floatToIntBits(bufColorF32Img[yp][xp].g)));
          data_out.writeInt(Integer.reverseBytes(Float.floatToIntBits(bufColorF32Img[yp][xp].b)));
        }
      }
    } catch (IOException exc) {
      exc.printStackTrace();
      Debug.stackTrace(exc);
    } finally {
      if (header_out != null)
        header_out.close();
      if (out != null)
        try {
          out.close();
        } catch (IOException exc) {
          exc.printStackTrace();
          Debug.stackTrace(exc);
        }
    }
  }

  public void writeRGB_JPG(File imageOutFile) {
    // see: https://www.universalwebservices.net/web-programming-resources/java/adjust-jpeg-image-compression-quality-when-saving-images-in-java/
    Iterator<ImageWriter> iter = javax.imageio.ImageIO.getImageWritersByFormatName("jpeg");
    javax.imageio.ImageWriter writer = iter.next();
    javax.imageio.ImageWriteParam params = writer.getDefaultWriteParam();
    // Compress using the compression type and quality settings specified in this ImageWriteParam.
    // Any previously set compression parameters are discarded.
    params.setCompressionMode(javax.imageio.ImageWriteParam.MODE_EXPLICIT);
    // quality is a float between 0.0f and 1.0f
    params.setCompressionQuality(1.0f);
    
    FileImageOutputStream output;
    try {
      output = new FileImageOutputStream(imageOutFile);
      writer.setOutput(output);
      javax.imageio.IIOImage image = new javax.imageio.IIOImage(bufAwtImg, null, null);
      writer.write(null, image, params);
      writer.dispose();
    } catch (IOException e) {
      e.printStackTrace();
    }    
  }
  /**
   * Method clone
   *
   * @return
   */
  public Object clone() {
    PixelCalculation calc = null;
    // !! try {
    calc = (PixelCalculation) super.clone();
    // !! } catch (CloneNotSupportedException exc) {
    // !! Debug.stackTrace(exc);
    // !! }
    return calc;
  }

}
