/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid.calculation;

import java.awt.Component;
import java.awt.Dimension;

import kandid.soup.ChromosomeType;
import kandid.Fitness;
import kandid.colorator.*;
import kandid.util.Debug;

/**
 * @author thomas jourdan
 */
public abstract class Calculation implements Cloneable {
  public static final int cmdSetBest      = 1;
  public static final int cmdResetBest    = 2;
  public static final int cmdSetAbort     = 3;
  public static final int cmdResetAbort   = 4;
  public static final int cmdSetParent1   = 5;
  public static final int cmdResetParent1 = 6;
  public static final int cmdSetParent2   = 7;
  public static final int cmdResetParent2 = 8;
  public static final int cmdResetParent  = 9;
  
  public ChromosomeType    chromosome;
  public Colorator     colorator;
  protected Fitness       fitness;
  public Dimension     canvasSize;
  protected Component viewComponent;
  protected boolean aborted;
  protected String imageFormat = "png";

  /**
   * Constructor Calculation
   */
  public Calculation() {
    fitness = new Fitness();
  }

  /**
   * Initialize the calculation with the given chromosome and colorator.
   *
   * @param chromosome Chromosome including genes.
   * @param colorator Strategy for converting the calculated results to color space.
   */
  public void setChromosome(ChromosomeType chromosome, Colorator colorator) {
    this.chromosome = chromosome;
    this.colorator = colorator;
  }
  
  /**
   * return the preferred size for this calculation.
   *
   * @return 
   */
  public Dimension getPreferredSize() {
//!!    if (Debug.enabled) assert canvasSize != null : "canvasSize is undefined";
//!!    return canvasSize;
    return null;
  }


  /**
   * set size of painting canvas
   *
   */
  public void activateCanvas(Component viewComponent, Dimension canvasSize, boolean zoomMode) {
    if(Debug.enabled) assert canvasSize.width > 0:  "canvasSize.width must be a positive integer";
    if(Debug.enabled) assert canvasSize.height > 0: "canvasSize.height must be a positive integer";
    this.viewComponent = viewComponent;
    this.canvasSize = canvasSize;
  }
  

  /**
   * Do the calculation.
   */
  public abstract void calculate(boolean paintOnScreen, String exportFilename);

  /**
   * Returns the percentage value for the done work on the calculation.
   *
   * @return value between 0 and 100.
   */
  public abstract int getPercent();

  /**
   * Test if the calculation ist completely finished.
   *
   * @return True, if the calculation is finished.
   */
  public abstract boolean getReady();

  /**
   * Process a fitness command.
   *
   * @param cmd
   */
  public void command(int cmd) {
    fitness.command(cmd);
  }

  /**
   * Returns the fitness of the chromosome used in this calculation.
   *
   * @return the fitness of the chromosome used in this calculation.
   */
  public Fitness getFitness() {
    return fitness;
  }

  /**
   * Sets fitness of the chromosome used in this calculation.
   *
   * @param fitness
   */
  public void setFitness(Fitness fitness) {
    this.fitness = fitness;
  }

  /**
   * Tests if this chromosome is able to produce a picture.
   * @param chromosome the chromosome to be tested.
   *
   * @return true if the calculation does not accept the chromosome.
   */
  public boolean reject(ChromosomeType chromosome) {
    return false;
  }

  /**
   * Describes the parameters of this calculation.
   *
   * @return parameter string.
   */
  protected String paramString() {
    return " width=" + canvasSize.width + " height=" + canvasSize.height + " " + chromosome.getClass().getName() + " " + colorator.getClass().getName() + " =fitness" + fitness;
  }

  /**
   * Returns a String representation of the calculation.
   *
   * @return a String representation of the calculation.
   */
  public String toString() {
    return getClass().getName() + "[" + paramString() + "]";
  }
  
  /**
   * rgb values used in this calculation.
   * returns null, if the number of colors is not a small and fix.
   *
   * @return null
   */
  public ColorF32[] getPalette() {
    return null;
  }

   /**
   * Method clone
   *
   * @return
   */
  public Object clone() {
    Calculation calc = null;
    try {
      calc = (Calculation)super.clone();
    } catch (CloneNotSupportedException exc) {
      Debug.stackTrace(exc);
    }
    return calc;
  }
  
  /**
   * Method getBackgroundColor.
   * @return int
   */
  public abstract boolean hasWhiteBackground();

  /**
   * returns true if an calculation is queed but deferred.
   */
  public boolean getDeferred() {
    return false;
  }

  /**
   * A <code>setAborted(true)</code> call will try to stop the calculation thread.
   * @param aborted
   */
  public void setAborted(boolean aborted) {
    this.aborted = aborted;
  }

  /**
   * @return true, if the calculation thread should give up.
   */
  public boolean isAborted() {
    return aborted;
  }

//!!  public void setGatway(Gateway gateway) {
//!!    this.gateway = gateway;
//!!  }

  /**
   * Set image format (png or jpg) for external image generators and outgoing mail.
   * @param imageFormat
   */
  public void setImgaeFormat(String imageFormat) {
    this.imageFormat = imageFormat;
  }

  /**
     * @return
     */
  public Component getViewComponent() {
    return viewComponent;
  }

}

