/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid;

import java.awt.*;
import javax.swing.*;

import kandid.pond.ui.IlPool;
import kandid.soup.*;
import kandid.calculation.*;
import kandid.colorator.*;
import kandid.util.*;

/**
 * Class ViewEngine
 * @author thomas jourdan
 * */
public class ViewEngine extends JLabel implements Runnable, Scrollable {

  private static final long serialVersionUID = 1L;
  private static final Dimension scrDim = Toolkit.getDefaultToolkit().getScreenSize();
  public static final int zoomWidth = 1024;
  private static final int zoomHeight = zoomWidth;
  
  private Dimension     dim;
  private boolean       zoomMode;
  private int           nRows;
  private int           nCols;
  private Thread        engine;    // Thread calculating refinements of the image
  private Object        sync = new Object();
  private Gateway       gateway;
  private boolean       ready;
  private OrganismPanel organismPanel;
  private IlPool ilPool;

  /**
   * Construct the panel
   *
   * @param zoomMode
   * @param nRows
   * @param nCols
   * @param organismPanel
   */
  public ViewEngine(int size, int nRows, int nCols, OrganismPanel organismPanel, IlPool ilPool) {
    this.zoomMode      = size > 200;
    this.nRows         = nRows;
    this.nCols         = nCols;
    this.organismPanel = organismPanel;
    this.ilPool = ilPool;
    gateway = new Gateway();
    calcDimension(size);
    this.setHorizontalAlignment(CENTER);
  }

  /**
   * Calculates an sets dimension for one image viewer.
   * @param size
   */
  private void calcDimension(int size) {
    if (size > 0) {
      dim = new Dimension(size, size);
    }
    else {
      int xl = (scrDim.width - (31 + 10)) / nCols;
      int yl = (scrDim.height - (nRows * 25 + 50)) / nRows;
      dim = (xl < yl)
            ? new Dimension(xl, xl)
            : new Dimension(yl, yl);
    }
    this.setPreferredSize(dim);
    this.setMaximumSize(dim);
    this.setMinimumSize(dim);
    this.setSize(dim);
  }

  /**
   * Method prepare
   *
   * @param chromosome
   * @param calculationName
   * @param colorator
   */
  public void prepare(ChromosomeType chromosome, Calculation calculation, Colorator colorator) {
    if(Debug.enabled) assert chromosome != null: "chromosome is undefined";
    if(Debug.enabled) assert calculation != null: "calculation is undefined";
    gateway.setChromosome(chromosome, calculation, colorator);
    if(zoomMode) {
      Dimension prefDim = gateway.getCalculation().getPreferredSize();
      if(prefDim != null) {
//        if(prefDim.width > zoomWidth)
//          prefDim.width = zoomWidth;
//        if(prefDim.height > zoomHeight)
//          prefDim.height = zoomHeight;
        if(prefDim.width < 125)
          prefDim.width = 125;
        if(prefDim.height < 125)
          prefDim.height = 125;
        dim = prefDim;
      }
      else {
        dim = new Dimension(zoomWidth, zoomHeight);
      }
    }
    this.setPreferredSize(dim);
    this.setMaximumSize(dim);
    this.setMinimumSize(dim);
    this.setSize(dim);
    gateway.activateScreenCanvas(dim, this, zoomMode);
    ImageIcon imageViewImageIcon = new ImageIcon(gateway.getImage());
    if (Debug.enabled) System.out.println("prepare.setIcon(imageViewImageIcon) " + imageViewImageIcon);
    setIcon(imageViewImageIcon);
  }
  
  /**
   * Start engine
   */
  public void start() {
    if (engine != null) {
      if(engine.isAlive()) {
      //!!      engine.stop();
        if (Debug.enabled) System.out.println("ViewEngine.interrupt() 1");
//!!        gw.getCalculation().setAborted(true);
        engine.interrupt();
      }
      engine = null;
    }
    if (engine == null) {
      if (Debug.enabled) System.out.println("ViewEngine.start()");
      engine = new Thread(this);
      engine.start();
    }
    if(organismPanel != null)
      organismPanel.restartAnimation();
  }

  /**
   * stop calculation
   */
  public void stop() {
    if (engine != null) {
      try {        
        if(engine.isAlive()) {
          if (Debug.enabled) System.out.println("ViewEngine.interrupt() 2");
          gateway.getCalculation().setAborted(true);
          engine.interrupt();
        }
      } catch (Throwable exc) {
        Debug.stackTrace(exc);
      }
      engine = null;
    }
    if(organismPanel != null) {
      organismPanel.hideStopButton();
      organismPanel.stopAnimation();
    }
  }

  /**
   * Method command
   *
   * @param cmd
   */
  public void command(int cmd) {
    gateway.getCalculation().command(cmd);
  }

  /**
   * Method getCalculation
   *
   * @return
   */
  public Calculation getCalculation() {
    return gateway.getCalculation();
  }

  /**
   * Method run
   */
  public void run() {
    long startTime = System.currentTimeMillis();
    Thread me = Thread.currentThread();
    // TODO GUI first
    //me.setPriority(Thread.NORM_PRIORITY + (zoomMode ? -1 : -2));
    //me.setPriority(Thread.NORM_PRIORITY + (zoomMode ? 2 : 1));
    me.setPriority(Thread.MIN_PRIORITY + (zoomMode ? 2 : 1));
//    if(gateway.getCalculation() instanceof PixelBridgeCalculation) {
//      ready = gateway.getReady();
//      while (!ready && (engine == me) && !Thread.currentThread().isInterrupted()) {
//        gateway.calculate(true, null);
//        gateway.updateImage(this);
//        ready = gateway.getReady();
//      }
//    }
//    else {
      while (!gateway.getReady() && (engine == me) && !Thread.currentThread().isInterrupted()) {
        ready = false;
        if(!gateway.getDeferred()) {
          Worker worker = new Worker(this);
          worker.start();
        }
        try {
          synchronized (sync) {
            while (!ready) {
              sync.wait();
            }
          }
        } catch (InterruptedException exc) {
          //!!Debug.stackTrace(exc);
          if(Debug.enabled) System.out.println("engine interrupted");
          return;
        }
        //!!       me.yield();
      }
//    }
    if(organismPanel != null) {
      organismPanel.hideStopButton();
      organismPanel.stopAnimation();
      organismPanel.getPopulationController().notifyReady(organismPanel.getPopulationID());
    }
    if(Debug.enabled) System.out.println("engine ready " + (System.currentTimeMillis()-startTime) + "ms");
  }

  /**
   * Class Worker
   * @author thomas jourdan
   */
  private class Worker extends SwingWorker {
    ViewEngine viewEngine;

    /**
     * Constructor for Worker.
     */
    public Worker(ViewEngine viewEngine) {
      super();
      this.viewEngine = viewEngine;
    }

    /**
     * Modify the values in the pixels array
     *
     * @return
     */
    public Object construct() {
      try {
        gateway.calculate(true, null);
      } catch (Throwable exc) {
        Debug.stackTrace(exc);
      }
      return gateway;
    }

    /**
     * Send the new data to the interested ImageConsumers
     */
    public void finished() {
      try {
        gateway.updateImage(viewEngine);
      } catch (Throwable exc) {
        Debug.stackTrace(exc);
      }
      synchronized (sync) {
        ready = true;
        sync.notify();
      }
    }
  }

  // --- Scrollable methods ---------------------------------------------

  /**
   * Returns the preferred size of the viewport for a view component.
   * This is implemented to do the default behavior of returning
   * the preferred size of the component.
   * 
   * @return the <code>preferredSize</code> of a <code>JViewport</code>
   * whose view is this <code>Scrollable</code>
   */
  public Dimension getPreferredScrollableViewportSize() {
    return getPreferredSize();
  }

  /**
   * Components that display logical rows or columns should compute
   * the scroll increment that will completely expose one new row
   * or column, depending on the value of orientation.  Ideally, 
   * components should handle a partially exposed row or column by 
   * returning the distance required to completely expose the item.
   * <p>
   * The default implementation of this is to simply return 10% of
   * the visible area.  Subclasses are likely to be able to provide
   * a much more reasonable value.
   * 
   * @param visibleRect the view area visible within the viewport
   * @param orientation either <code>SwingConstants.VERTICAL</code> or
   *   <code>SwingConstants.HORIZONTAL</code>
   * @param direction less than zero to scroll up/left, greater than
   *   zero for down/right
   * @return the "unit" increment for scrolling in the specified direction
   * @exception IllegalArgumentException for an invalid orientation
   * @see JScrollBar#setUnitIncrement
   */
  public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction) {
    switch (orientation) {
      case SwingConstants.VERTICAL :
        return visibleRect.height / 10;
      case SwingConstants.HORIZONTAL :
        return visibleRect.width / 10;
      default :
        throw new IllegalArgumentException("Invalid orientation: " + orientation);
    }
  }

  /**
   * Components that display logical rows or columns should compute
   * the scroll increment that will completely expose one block
   * of rows or columns, depending on the value of orientation. 
   * <p>
   * The default implementation of this is to simply return the visible
   * area.  Subclasses will likely be able to provide a much more 
   * reasonable value.
   * 
   * @param visibleRect the view area visible within the viewport
   * @param orientation either <code>SwingConstants.VERTICAL</code> or
   *   <code>SwingConstants.HORIZONTAL</code>
   * @param direction less than zero to scroll up/left, greater than zero
   *  for down/right
   * @return the "block" increment for scrolling in the specified direction
   * @exception IllegalArgumentException for an invalid orientation
   * @see JScrollBar#setBlockIncrement
   */
  public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction) {
    switch (orientation) {
      case SwingConstants.VERTICAL :
        return visibleRect.height;
      case SwingConstants.HORIZONTAL :
        return visibleRect.width;
      default :
        throw new IllegalArgumentException("Invalid orientation: " + orientation);
    }
  }

  /**
   * Returns true if a viewport should always force the width of this 
   * <code>Scrollable</code> to match the width of the viewport.
   * For example a normal text view that supported line wrapping
   * would return true here, since it would be undesirable for
   * wrapped lines to disappear beyond the right
   * edge of the viewport.  Note that returning true for a
   * <code>Scrollable</code> whose ancestor is a <code>JScrollPane</code>
   * effectively disables horizontal scrolling.
   * <p>
   * Scrolling containers, like <code>JViewport</code>,
   * will use this method each time they are validated.  
   * 
   * @return true if a viewport should force the <code>Scrollable</code>s
   *   width to match its own
   */
  public boolean getScrollableTracksViewportWidth() {
    if (getParent() instanceof JViewport) {
      return (((JViewport) getParent()).getWidth() > getPreferredSize().width);
    }
    return false;
  }

  /**
   * Returns true if a viewport should always force the height of this 
   * <code>Scrollable</code> to match the height of the viewport.
   * For example a columnar text view that flowed text in left to
   * right columns could effectively disable vertical scrolling by 
   * returning true here.
   * <p>
   * Scrolling containers, like <code>JViewport</code>,
   * will use this method each time they are validated.  
   * 
   * @return true if a viewport should force the Scrollables height
   *   to match its own
   */
  public boolean getScrollableTracksViewportHeight() {
    if (getParent() instanceof JViewport) {
      return (((JViewport) getParent()).getHeight() > getPreferredSize().height);
    }
    return false;
  }

  /**
   * Returns the organismPanel.
   * @return OrganismPanel
   */
  public OrganismPanel getOrganismPanel() {
    return organismPanel;
  }

  /**
   * @return
   */
  public IlPool getPoolController() {
    return ilPool;
  }

  /**
   * Returns the zoomMode.
   * @return boolean
   */
  public boolean isZoomMode() {
    return zoomMode;
  }

  /**
   * @return
   */
  public Gateway getGateway() {
    return gateway;
  }

  /**
   * @return
   */
  public Dimension getDim() {
    return dim;
  }

}
