/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid;

import java.awt.*;
import java.awt.event.*;

import javax.swing.*;

import kandid.soup.*;
import kandid.soup.util.SoupFactory;
import kandid.calculation.*;
import kandid.colorator.*;
import kandid.extensions.*;
import kandid.util.*;

/**
 * Class OrganismPanel
 * @author thomas jourdan
 * */
public class OrganismPanel extends JPanel implements DataSink {
  private static final long serialVersionUID = 1L;
  private ViewEngine            viewEngine;
  private SoupFactory           soupFactory;
  private PopulationController  populationController;
  private int                   populationID;
  private Thread                percentAnimation;
  private ChromosomeEditor      chromosomeEditor;
  
  BorderLayout   borderLayout1      = new BorderLayout();
  JComponent     imageComponent;
  JPanel         organismPanel      = new JPanel();
  BorderLayout   borderLayout2      = new BorderLayout();
  JPanel         headlinePanel      = new JPanel();
  JLabel         percentageLabel    = new JLabel();
  JButton        stopButton         = new JButton();
  JToggleButton  bestButton         = new JToggleButton();
  JToggleButton  abortButton        = new JToggleButton();
  JToggleButton  parent1Button      = new JToggleButton();
  JToggleButton  parent2Button      = new JToggleButton();
  JButton        zoomButton         = new JButton();
  JButton        exportButton       = new JButton();
  JButton        editButton         = new JButton();
  JPanel         cntrlPanel         = new JPanel();
  FlowLayout     cntrlFlowLayout    = new FlowLayout();
  FlowLayout     headlineFlowLayout = new FlowLayout();
  Font smallFont = new Font("Dialog", Font.PLAIN, 8);

  /**
   * Constructor OrganismPanel
   *
   * @param zoomMode
   * @param nRows
   * @param nCols
   * @param populationController
   * @param populationID
   */
  public OrganismPanel(boolean zoomMode, int nRows, int nCols, boolean enableScrollbar, PopulationController populationController, int populationID) {
    super(true);
    this.populationController = populationController;
    this.populationID        = populationID;
    viewEngine = new ViewEngine(zoomMode ? ViewEngine.zoomWidth : 0, nRows, nCols, this, null);
    soupFactory = SoupFactory.getSoupFactory();
    try {
      if (zoomMode && enableScrollbar) {
        imageComponent = new JPanel();
        imageComponent.setLayout(new BorderLayout());
        JScrollPane jScrollPane = new JScrollPane(viewEngine);
        imageComponent.add(jScrollPane, BorderLayout.CENTER);
      }
      else if (zoomMode) {
        imageComponent = viewEngine;
      }
      else {
        imageComponent = viewEngine;
        // drag and drop
        MouseListener listener = new DragMouseAdapter();
        viewEngine.addMouseListener(listener);
        viewEngine.setTransferHandler(new ImageTransferHandler(false));
      }
      imageComponent.setDoubleBuffered(true);
      
      jbInit();
      if (zoomMode || populationController == null) {
        bestButton.setEnabled(false);
        abortButton.setEnabled(false);
        parent1Button.setEnabled(false);
        parent2Button.setEnabled(false);
        zoomButton.setEnabled(false);
        editButton.setEnabled(false);
      }
    }
    catch (Throwable exc) {
      Debug.stackTrace(exc);
    }
  }

  /**
   * Method adjustPanelSize
   */
  public void adjustPanelSize() {
    Dimension ivDim = viewEngine.getPreferredSize();
    Dimension dim   = new Dimension((ivDim.width < 125)
                                  ? 125
                                  : ivDim.width, ivDim.height + 17);
    this.setPreferredSize(dim);
    this.setMaximumSize(dim);
    this.setMinimumSize(dim);
  }

  /**
   * Method jbInit
   *
   * @throws Exception
   */
  private void jbInit() throws Exception {
    this.setLayout(borderLayout1);
    this.setBorder(BorderFactory.createEtchedBorder());
    organismPanel.setLayout(borderLayout2);
    percentageLabel.setText("");
//  !!    stopButton.setText("");
    
    stopButton.setMargin(new Insets(0, 0, 0, 0));
    stopButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        stopButton_actionPerformed(e);
      }
    });
    stopButton.setActionCommand("stopButton");
    stopButton.setIcon(IconLoader.stopImageIcon);
    stopButton.setToolTipText("stop calculation");
    stopButton.setFont(smallFont);


    bestButton.setMargin(new Insets(0, 0, 0, 0));
    bestButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        bestButton_actionPerformed(e);
      }
    });
    bestButton.setActionCommand("bestButton");
    bestButton.setIcon(IconLoader.bestImageIcon);
    bestButton.setToolTipText("mark this as best picture");
    bestButton.setSelectedIcon(IconLoader.bestSelectedImageIcon);
    abortButton.setMargin(new Insets(0, 0, 0, 0));
    abortButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        abortButton_actionPerformed(e);
      }
    });
    abortButton.setActionCommand("abortButton");
    abortButton.setIcon(IconLoader.abortImageIcon);
    abortButton.setToolTipText("mark this picture for deletion");
    abortButton.setSelectedIcon(IconLoader.abortSelectedImageIcon);
    parent1Button.setMargin(new Insets(0, 0, 0, 0));
    parent1Button.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        parent1Button_actionPerformed(e);
      }
    });
    parent1Button.setActionCommand("parent1Button");
    parent1Button.setIcon(IconLoader.parent1ImageIcon);
    parent1Button.setSelectedIcon(IconLoader.parent1SelectedImageIcon);
    parent1Button.setToolTipText("mark this picture as first parent");
    parent2Button.setMargin(new Insets(0, 0, 0, 0));
    parent2Button.setIcon(IconLoader.parent2ImageIcon);
    parent2Button.setSelectedIcon(IconLoader.parent2SelectedImageIcon);
    parent2Button.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        parent2Button_actionPerformed(e);
      }
    });
    parent2Button.setToolTipText("mark this picture to be the other parent");
    zoomButton.setMargin(new Insets(0, 0, 0, 0));
    zoomButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        zoomButton_actionPerformed(e);
      }
    });
    zoomButton.setActionCommand("zoomButton");
    zoomButton.setIcon(IconLoader.zoomImageIcon);
    zoomButton.setToolTipText("zoom picture");

    exportButton.setMargin(new Insets(0, 0, 0, 0));
    exportButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        exportButton_actionPerformed(e);
      }
    });
    exportButton.setActionCommand("exportButton");
    exportButton.setIcon(IconLoader.exportImageIcon);
      exportButton.setToolTipText("export picture");

    editButton.setMargin(new Insets(0, 0, 0, 0));
    editButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        editButton_actionPerformed(e);
      }
    });
    editButton.setActionCommand("editButton");
    editButton.setIcon(IconLoader.editImageIcon);
    editButton.setToolTipText("edit chromosome");
    
    cntrlPanel.setLayout(cntrlFlowLayout);
    cntrlFlowLayout.setAlignment(FlowLayout.RIGHT);
    cntrlFlowLayout.setHgap(1);
    cntrlFlowLayout.setVgap(1);
    
    headlinePanel.setLayout(headlineFlowLayout);
    headlineFlowLayout.setAlignment(FlowLayout.LEFT);
    headlineFlowLayout.setHgap(1);
    headlineFlowLayout.setVgap(1);
    
    this.add(imageComponent, BorderLayout.CENTER);
    this.add(organismPanel, BorderLayout.NORTH);
    organismPanel.add(headlinePanel, BorderLayout.WEST);
    headlinePanel.add(stopButton, null);
    headlinePanel.add(percentageLabel, null);
    organismPanel.add(cntrlPanel, BorderLayout.EAST);
    
    cntrlPanel.add(zoomButton, null);
    cntrlPanel.add(editButton, null);
    cntrlPanel.add(exportButton, null);
    cntrlPanel.add(parent1Button, null);
    cntrlPanel.add(parent2Button, null);
    cntrlPanel.add(abortButton, null);
    cntrlPanel.add(bestButton, null);
  }

  /**
   * @param e
   */
  protected void stopButton_actionPerformed(ActionEvent e) {
    Calculation calc = getCalculation();
    calc.setAborted(true);
  }

  /**
   * Method bestButton_actionPerformed
   *
   * @param e
   */
  void bestButton_actionPerformed(ActionEvent e) {
    if (populationController != null) {
      populationController.dispatchBest(populationID, bestButton.isSelected());
    }
  }

  /**
   * Method abortButton_actionPerformed
   *
   * @param e
   */
  void abortButton_actionPerformed(ActionEvent e) {
    if (populationController != null) {
      populationController.dispatchAbort(populationID, abortButton.isSelected());
    }
  }

  /**
   * Method parent1Button_actionPerformed
   *
   * @param e
   */
  void parent1Button_actionPerformed(ActionEvent e) {
    if (populationController != null) {
      populationController.dispatchParent1(populationID, parent1Button.isSelected());
    }
  }

  /**
   * Method parent2Button_actionPerformed
   *
   * @param e
   */
  void parent2Button_actionPerformed(ActionEvent e) {
    if (populationController != null) {
      populationController.dispatchParent2(populationID, parent2Button.isSelected());
    }
  }

  /**
   * Method exportButton_actionPerformed
   *
   * @param e
   */
  void exportButton_actionPerformed(ActionEvent e) {
    ExportWizard exportFrame = new ExportWizard(populationController.getPopulation(), populationID, getCalculation(), populationController.getTitle(populationID));
    exportFrame.setTitle("Export: " + populationController.getTitle(populationID));
    exportFrame.pack();
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    exportFrame.setLocation((d.width - exportFrame.getSize().width) / 2, (d.height - exportFrame.getSize().height) / 2);
    exportFrame.pack();
    exportFrame.setVisible(true);
  }

  /**
   * Method editButton_actionPerformed
   *
   * @param e
   */
  void editButton_actionPerformed(ActionEvent e) {
    if(chromosomeEditor == null) {
      PopulationType population = populationController.getPopulation();
      ChromosomeType chromosome = soupFactory.getChromosome(population, populationID);
      ColoratorType colorator = soupFactory.getColorator(population, populationID);
      chromosomeEditor = new ChromosomeEditor(this, "Edit: " + populationController.getTitle(populationID));
      soupFactory.fillEditor(chromosome, colorator, chromosomeEditor.getTree());
      chromosomeEditor.setSize(300, 500);
      chromosomeEditor.setVisible(true);
    }
    else {
      chromosomeEditor.toFront();
    }
  }

  /**
   * Method zoomButton_actionPerformed
   *
   * @param e
   */
  void zoomButton_actionPerformed(ActionEvent e) {
    Calculation calc = getCalculation();
    PopulationType population = populationController.getPopulation();
    ZoomFrame zoomFrame = new ZoomFrame(populationController);
    zoomFrame.setTitle("Zoom: " + populationController.getTitle(populationID));
    zoomFrame.pack();
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    zoomFrame.setLocation((d.width - zoomFrame.getSize().width) / 2, (d.height - zoomFrame.getSize().height) / 2);
    zoomFrame.setVisible(true);
    zoomFrame.prepare(population, populationID, calc);
  }

  /**
   * start calculation
   *
   * @param chromosome
   * @param calculationName
   * @param colorator
   */
  private void start(ChromosomeType chromosome, Calculation calculation, Colorator colorator, Fitness fitness) {
    percentageLabel.setText("");
//  !!    stopButton.setText("");
    stopButton.setEnabled(true);
    stopButton.setVisible(true);
    try {
      viewEngine.prepare(chromosome, calculation, colorator);
      viewEngine.start();
    } catch (Exception e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    calculation.setFitness(fitness);
    toggleButtons();
  }

  public void start(PopulationType population, int nx, Fitness fitness) {
    try {
      this.populationID        = nx;
      ChromosomeType chromosome = soupFactory.getChromosome(population, nx);
      Calculation calculation = (new Gateway()).createCalculation(soupFactory.getCalculationName(population, nx));
      ColoratorType coloratorType = soupFactory.getColorator(population, nx);
      String coloratorName = soupFactory.getColoratorName(population, nx);
      Colorator colorator = ColoratorFactory.createColoratorWorker(coloratorName, coloratorType);
      start(chromosome, calculation, colorator, fitness);
    } catch (Exception exc) {
      Debug.stackTrace(exc);
    }
  }
  
  /**
   * Method startAnimation
   */
  public void startAnimation() {
    if (percentAnimation == null) {
      percentAnimation = new Animation();
      percentAnimation.start();
    }
  }

  /**
   * Method restartAnimation
   */
  public void restartAnimation() {
    if (percentAnimation != null) {
      percentAnimation.interrupt();
      percentAnimation = null;
    }
    startAnimation();
  }

  /**
   * stop calculation
   */
  public void stop() {
    if(chromosomeEditor != null) {
      chromosomeEditor.dispose();
      chromosomeEditor = null;
    }
    stopButton.setEnabled(false);
    stopButton.setVisible(false);
    viewEngine.stop();
  }

  /**
   * Method stopAnimation
   */
  public void stopAnimation() {
    percentAnimation = null;
  }

  /**
   * Method getCalculation
   *
   * @return
   */
  public Calculation getCalculation() {
    return viewEngine.getCalculation();
  }

  /**
   * Method command
   *
   * @param cmd
   */
  public void command(int cmd) {
    viewEngine.command(cmd);
  }

  /**
   * Method toggleButtons
   */
  public void toggleButtons() {
    Fitness fitness = viewEngine.getCalculation().getFitness();
    bestButton.setSelected(fitness.best);
    abortButton.setSelected(fitness.abort);
    parent1Button.setSelected(fitness.parent1);
    parent2Button.setSelected(fitness.parent2);
  }

  /**
   * implements DataSink.quit
   */
  public void quit() {
    chromosomeEditor = null;
  }

  /**
   * implements DataSink.apply
   */
  public void apply() {
    PopulationType population = populationController.getPopulation();
    ChromosomeType chromosome = soupFactory.getChromosome(population, populationID);
    LRUImageCache.getLRUImageCache().removeAll(chromosome.getIdent());
    populationController.apply(populationID);
  }


  /**
   * Class Animation
   * @author thomas jourdan
   */
  private class Animation extends Thread {

    /**
     * Method run
     */
    public void run() {
      Thread me = Thread.currentThread();
      me.setPriority(Thread.NORM_PRIORITY+1);
      percentageLabel.setForeground(Color.red);
      stopButton.setForeground(Color.red);
      while (!viewEngine.getCalculation().getReady() && (percentAnimation == me)) {
        try {
          int percent = viewEngine.getCalculation().getPercent();
          if(percent >= 0) {
            percentageLabel.setText(""+viewEngine.getCalculation().getPercent());
//          !!            stopButton.setText(viewEngine.getCalculation().getPercent() + "%");
          }
          else {
            percentageLabel.setText("");
//          !!            percentageLabel.setText("?%");
//          !!            stopButton.setText("?%");
          }
          headlinePanel.repaint(); //!!
          Thread.sleep(600);
        } catch (InterruptedException exc) {
          percentAnimation = null;
        } catch (Exception exc) {
          Debug.stackTrace(exc);
        }
      }
      percentageLabel.setText("");
//    !!      stopButton.setText("");
      hideStopButton();
      percentAnimation = null;
    }
  }
  
  /**
   * Returns the populationController.
   * @return PopulationController
   */
  public PopulationController getPopulationController() {
    return populationController;
  }

  /**
   * Returns the populationID.
   * @return int
   */
  public int getPopulationID() {
    return populationID;
  }


  private class DragMouseAdapter extends MouseAdapter {
    public void mousePressed(MouseEvent e) {
      JComponent c = (JComponent) e.getSource();
      TransferHandler handler = c.getTransferHandler();
      handler.exportAsDrag(c, e, TransferHandler.COPY);
    }
  }
  
  /**
   * @return
   */
  public ViewEngine getViewEngine() {
    return viewEngine;
  }

  /**
   * 
   */
  public void hideStopButton() {
    stopButton.setEnabled(false);
    stopButton.setVisible(false);
  }

}
