/*
 * Copyright (C) 2002 - 2025 Thomas Jourdan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

package kandid;

import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.DefaultMutableTreeNode;
import java.awt.BorderLayout;
import java.awt.event.ActionEvent;

import javax.swing.*;

import kandid.tree.renderer.*;
import kandid.tree.editor.*;

import kandid.soup.map.SetEditorMapping;

public class ChromosomeEditor extends javax.swing.JFrame {
  private static final long serialVersionUID = 1L;

  private static ImageIcon cancelImageIcon;
  private static ImageIcon acceptImageIcon;

  private JTree         tree;
  private String        title;
  private DataSink      caller;
  private JScrollPane   jScrollPane1 = new javax.swing.JScrollPane();
  private JButton       okButton     = new JButton("Ok");
  private JButton       applyButton  = new JButton("Apply");
  private JButton       cancelButton = new JButton("Close");
  private JPanel        footerPanel  = new JPanel();

  /**
   * @see java.awt.Frame#Frame(String)
   */
  public ChromosomeEditor(DataSink caller, String title) {
    this.caller = caller;
    this.title = title;
    initComponents();
    getRootPane().setDefaultButton(applyButton);
    okButton.setVisible(false); //!!
  }

  /**
   * Method initComponents.
   * @param title
   */
  private void initComponents() {
    setTitle(title);
    addWindowListener(new java.awt.event.WindowAdapter() {
      public void windowClosing(java.awt.event.WindowEvent evt) {
        closeEditor();
      }
    });

    tree = new javax.swing.JTree();
    tree.setEditable(true);
    tree.setModel(createModel());
    jScrollPane1.setViewportView(tree);
    getContentPane().add(jScrollPane1, java.awt.BorderLayout.CENTER);

    if (acceptImageIcon == null)
      acceptImageIcon = Kandid.getInstance().createImageIcon("accept.png", null);
    okButton.setIcon(acceptImageIcon);
    okButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        okButton_actionPerformed(e);
      }
    });
    applyButton.setIcon(acceptImageIcon);
    applyButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        applyButton_actionPerformed(e);
      }
    });
    if (cancelImageIcon == null)
      cancelImageIcon = Kandid.getInstance().createImageIcon("cancel.png", null);
    cancelButton.setIcon(cancelImageIcon);
    cancelButton.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        cancelButton_actionPerformed(e);
      }
    });
    footerPanel.add(okButton, null);
    footerPanel.add(applyButton, null);
    footerPanel.add(cancelButton, null);
    getContentPane().add(footerPanel, BorderLayout.SOUTH);

    pack();
  }

  /**
   * Method okButton_actionPerformed
   *
   * @param e
   */
  void okButton_actionPerformed(ActionEvent e) {
    if (caller != null) {
      caller.apply();
    }
    closeEditor();
  }

  /**
   * Method applyButton_actionPerformed
   *
   * @param e
   */
  void applyButton_actionPerformed(ActionEvent e) {
    if (caller != null) {
      caller.apply();
    }
  }

  /**
   * Method cancelButton_actionPerformed
   *
   * @param e
   */
  void cancelButton_actionPerformed(ActionEvent e) {
    closeEditor();
  }

  /**
   * Method closeEditor
   */
  private void closeEditor() {
    if (caller != null) {
      caller.quit();
    }
    dispose();
    SetEditorMapping.dump(tree);
  }

  /**
   * Method createModel.
   * @return DefaultTreeModel
   */
  private DefaultTreeModel createModel() {
    UserNodeRenderer renderer = new UserNodeRenderer();
    UserNodeEditor editor = new UserNodeEditor(tree, renderer, null);
    tree.setCellRenderer(renderer);
    tree.setCellEditor(editor);

    DefaultMutableTreeNode root = new DefaultMutableTreeNode(title);
    return new DefaultTreeModel(root);
  }

  /**
   * Returns the tree.
   * @return javax.swing.JTree
   */
  public javax.swing.JTree getTree() {
    return tree;
  }

  /**
   * Sets the tree.
   * @param tree The tree to set
   */
  public void setTree(javax.swing.JTree tree) {
    this.tree = tree;
  }

}
